unit UFileUpdater;

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants,
  System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, SyncObjs, SHFolder, Vcl.StdCtrls,
  Vcl.ComCtrls, ShlObj, ComObj, activex;

type
  TUpdateFile = record
    SourceFileName: String;
    DestFileName: String;
  end;

  tTUpdateFiles = Array of TUpdateFile;

  TfrmUpdate = class;

  // Thread
  tUpdateThread = class(TThread)
  private
    fSuccess: Boolean;
    fDone: Boolean;
    fForm: TfrmUpdate;
    fMsgToUse: String;
    fSourceFolder: String;
    fDestFolder: String;
    fUpdatFileList: tTUpdateFiles;
    fErrOcur: Boolean;
    fErrMsg: String;
    fCriticalSec: TcriticalSection;
    fUpdateNeeded: Boolean;
    fProgMax: Integer;
    fProgPos: Integer;
    procedure DoExecute;
    procedure DoFinish;
    Procedure UpdateProgressBar;
    Procedure SetupProgCheck;
    Procedure SetupProgDown;
    procedure BuildFileList;
    procedure DownLoadFiles();
    Function IsUpdateNeeded: Boolean;
    procedure AddFileForUpdate(const AFileName, DestFileName: String);
    function getFileDateTime(const FileName: string): TDateTime;
    function GetDestFolder: String;
    function GetSourceFolder: String;
    procedure SetDestFolder(const Value: String);
    procedure SetSourceFolder(const Value: String);
  protected
    procedure Execute; override;
  public
    constructor Create();
    destructor Destroy; override;
    Property HostForm: TfrmUpdate read fForm write fForm;
    Property ErrorOccured: Boolean read fErrOcur write fErrOcur;
    Property ErrorMessage: String read fErrMsg write fErrMsg;
    Property DestFolder: String read GetDestFolder write SetDestFolder;
    Property SourceFolder: String read GetSourceFolder write SetSourceFolder;
  end;

  TfrmUpdate = class(TForm)
    TaskText: TStaticText;
    ProgressBar1: TProgressBar;
    procedure FormCreate(Sender: TObject);
  protected
   procedure CreateParams(var Params: TCreateParams); override;
  private
    { Private declarations }
    fTheThread: tUpdateThread;
    fSourceFolder: String;
    fDestFolder: String;
    fThreadErrorOccured: Boolean;
    fThreadErrorMessage: String;
    fisWin7: Boolean;
    _TaskBarList: ITaskbarList;
    _TaskbarList3: ITaskbarList3;
    _TaskbarHandle: HWnd;
    function GetDestFolder: String;
    procedure SetDestFolder(const Value: String);
    function GetSourceFolder: String;
    procedure SetSourceFolder(const Value: String);
    procedure StartFileTransfer();
  public
    { Public declarations }
    Property aTaskBarList: ITaskbarList read _TaskBarList write _TaskBarList;
    Property aTaskbarList3: ITaskbarList3 read _TaskbarList3 write _TaskbarList3;
    Property aTaskbarHandle: HWnd read _TaskbarHandle write _TaskbarHandle;
    Property DestFolder: String read GetDestFolder write SetDestFolder;
    Property IsWin7: Boolean read fisWin7;
    Property SourceFolder: String read GetSourceFolder write SetSourceFolder;
    Property ThreadErrorOccured: Boolean read fThreadErrorOccured write fThreadErrorOccured;
    Property ThreadErrorMessage: String read fThreadErrorMessage write fThreadErrorMessage;
  end;

Function UpdateFiles(var DestFolder: String): Boolean;

implementation

uses
  System.Generics.Collections, System.IOUtils, Masks, Shellapi, system.Types, system.UITypes;

Const
  fFileMask: {Array [0..0] of string = ('*.*');}Array [0 .. 3] of string = ('*.mdb', '*.txml', '*.err', '*.hsm');
  fDownloadMsg = 'Updating %s out of %s files';
  fCheckMsg = 'Checking for updates, Please wait ... (%s of %s)';
  fTaskProg = true;

{$R *.dfm}

Function UpdateFiles(var DestFolder: String): Boolean;
var
  frmUpdate: TfrmUpdate;
begin
  Result := false;
  frmUpdate := TfrmUpdate.Create(nil);
  try
    frmUpdate.StartFileTransfer;
    if fTaskProg and frmUpdate.IsWin7 then
    begin
     frmUpdate.WindowState := wsMinimized;
     frmUpdate.ProgressBar1.Visible := false;
    end;
    frmUpdate.ShowModal;
    DestFolder := frmUpdate.DestFolder;

    if frmUpdate.ThreadErrorOccured then
    begin
      Result := true;
      TaskMessageDlg('Error occured',
        'An error occurred during file transmission. ' +
        'Please try again and if the problem persist contact support' + #13#10 +
        'Error: ' + frmUpdate.ThreadErrorMessage, mtError, mbYesNo, -1);
    end;
  finally
    frmUpdate.Free;
  end;
end;

{$REGION 'tUpdateThread'}

constructor tUpdateThread.Create();
begin
  inherited Create(true);
  fSuccess := false;
  fDone := false;
  SetLength(fUpdatFileList, 0);
  fCriticalSec := TcriticalSection.Create;
  FreeOnTerminate := true;
end;

destructor tUpdateThread.Destroy;
begin
  fDone := true;
  Synchronize(DoFinish);
  SetLength(fUpdatFileList, 0);
  FreeAndNil(fCriticalSec);
  inherited;
end;

procedure tUpdateThread.Execute;
begin
  if Terminated then
    Exit;

  Sleep(Random(100));

  DoExecute;

  Sleep(Random(100));

  fDone := true;

end;

function tUpdateThread.GetDestFolder: String;
begin
  Result := fDestFolder;
end;

function tUpdateThread.GetSourceFolder: String;
begin
  Result := fSourceFolder;
end;

procedure tUpdateThread.SetDestFolder(const Value: String);
begin
  fDestFolder := Value;
end;

procedure tUpdateThread.SetSourceFolder(const Value: String);
begin
  fSourceFolder := Value;
end;


procedure tUpdateThread.SetupProgCheck;
begin
  if Assigned(HostForm) then
  begin
  if HostForm.IsWin7 and fTaskProg then
  begin
    try
      if Assigned(HostForm.aTaskbarList3) then
      begin
        if HostForm.aTaskbarHandle <> 0 then
        begin
          fProgMax := Length(TDirectory.GetFiles(fSourceFolder));
          fProgPos := 0;
          HostForm.aTaskbarList3.SetProgressValue(HostForm.aTaskbarHandle, fProgPos, fProgMax);
        end;
      end;
    except
      HostForm.aTaskbarList3 := nil;
      HostForm.aTaskBarList := nil;
      HostForm.aTaskbarHandle := 0;
    end;
  end else
  begin
    fMsgToUse := fCheckMsg;
    HostForm.ProgressBar1.Position := 0;
    HostForm.ProgressBar1.Max := Length(TDirectory.GetFiles(fSourceFolder));
    HostForm.TaskText.Caption := Format(fMsgToUse,
      ['0', IntToStr(HostForm.ProgressBar1.Max)]);
    end;
  end;
end;

procedure tUpdateThread.SetupProgDown;
begin
    if HostForm.IsWin7 and fTaskProg then
  begin
    try
      if Assigned(HostForm.aTaskbarList3) then
      begin
        if HostForm.aTaskbarHandle <> 0 then
        begin
          fProgMax := Length(fUpdatFileList);
          fProgPos := 0;
          HostForm.aTaskbarList3.SetProgressValue(HostForm.aTaskbarHandle, fProgPos, fProgMax);
        end;
      end;
    except
      HostForm.aTaskbarList3 := nil;
      HostForm.aTaskBarList := nil;
      HostForm.aTaskbarHandle := 0;
    end;
  end else
  begin
  fMsgToUse := fDownloadMsg;

  HostForm.ProgressBar1.Position := 0;
  HostForm.ProgressBar1.Max := Length(fUpdatFileList);
  HostForm.TaskText.Caption := Format(fMsgToUse,
    ['0', IntToStr(HostForm.ProgressBar1.Max)]);
  end;
end;

procedure tUpdateThread.UpdateProgressBar;
var
  ProgPos, ProgMax: Integer;
begin
     if HostForm.IsWin7 and fTaskProg then
  begin
    try
      if Assigned(HostForm.aTaskbarList3) then
      begin
        if HostForm.aTaskbarHandle <> 0 then
        begin
          fProgPos := fProgPos + 1;
          HostForm.aTaskbarList3.SetProgressValue(HostForm.aTaskbarHandle, fProgPos, fProgMax);
        end;
      end;
    except
      HostForm.aTaskbarList3 := nil;
      HostForm.aTaskBarList := nil;
      HostForm.aTaskbarHandle := 0;
    end;
  end else
  begin
  ProgPos := HostForm.ProgressBar1.Position;
  ProgMax := HostForm.ProgressBar1.Max;

  HostForm.ProgressBar1.Position := ProgPos + 1;
  HostForm.TaskText.Caption := Format(fMsgToUse,
    [IntToStr(ProgPos), IntToStr(ProgMax)]);
  end;
end;

procedure tUpdateThread.DoExecute;
begin
  // Start the check
  if IsUpdateNeeded then
  begin
    // Start the download
    DownLoadFiles;
  end;

end;

procedure tUpdateThread.DoFinish;
begin
  // sync to set the dest folder
  HostForm.ThreadErrorOccured := fErrOcur;
  HostForm.ThreadErrorMessage := fErrMsg;
  if Assigned(HostForm) then
    HostForm.ModalResult := mrClose;

  if HostForm.IsWin7 and fTaskProg then
  begin
  try
        if Assigned(HostForm.aTaskbarList3) then
        begin
          if HostForm.aTaskbarHandle <> 0 then
            HostForm.aTaskbarList3.SetProgressState(HostForm.aTaskbarHandle, TBPF_NOPROGRESS);
        end;
      except
        // do nothing
      end;
      HostForm.aTaskbarList3 := nil;
      HostForm.aTaskBarList := nil;
      HostForm.aTaskbarHandle := 0;
  end;
end;

procedure tUpdateThread.AddFileForUpdate(const AFileName, DestFileName: String);

  function GetFileSize(const AFileName: String): Int64;
  var
    fileinfo: TWin32FileAttributeData;
  begin
    Result := -1;
    if NOT GetFileAttributesEx(PWideChar(AFileName), GetFileExInfoStandard,
      @fileinfo) then
      Exit;
    Result := Int64(fileinfo.nFileSizeLow) or
      Int64(fileinfo.nFileSizeHigh shl 32);
  end;

begin
  try
    SetLength(fUpdatFileList, Length(fUpdatFileList) + 1);
    fUpdatFileList[High(fUpdatFileList)].SourceFileName := AFileName;
    fUpdatFileList[High(fUpdatFileList)].DestFileName := DestFileName;
  Except
    On e: exception do
    begin
      fErrOcur := true;
      fErrMsg := e.Message;
    end;
  end;
end;

procedure tUpdateThread.BuildFileList;
var
  UpdNeeded: Boolean;
  AFileName, TmpDest: string;
  Predicate: TDirectory.TFilterPredicate;
begin
  try
    Predicate :=
        function(const Path: string; const SearchRec: TSearchRec): Boolean
      var
        Mask: string;
      begin
        for Mask in fFileMask do
          if MatchesMask(SearchRec.Name, Mask) then
            Exit(true);
        Exit(false);
      end;
    for AFileName in TDirectory.GetFiles(fSourceFolder,
      TSearchOption.soAllDirectories, Predicate) do
    begin
      TmpDest := StringReplace(AFileName, fSourceFolder, fDestFolder,
        [rfReplaceAll]);
      if not FileExists(TmpDest) then
        UpdNeeded := true
      else
      begin
        UpdNeeded := (getFileDateTime(AFileName) > getFileDateTime(TmpDest));
        // or (GetFileSize(aFileName) <> GetFileSize(destFolder));
      end;

      if UpdNeeded then
        AddFileForUpdate(AFileName, TmpDest);
      // Notify that we checked a file
      Synchronize(UpdateProgressBar);
    end;
  Except
    On e: exception do
    begin
      fErrOcur := true;
      fErrMsg := e.Message;
    end;
  end;
end;

procedure tUpdateThread.DownLoadFiles();
var
  I: Integer;
begin
  try

    try
      // Clear the progressbar
      Synchronize(SetupProgDown);

      for I := Low(fUpdatFileList) to High(fUpdatFileList) do
      begin
        if not ForceDirectories(ExtractFilePath(fUpdatFileList[I].DestFileName))
        then
          break;

        CopyFile(PChar(fUpdatFileList[I].SourceFileName),
          PChar(fUpdatFileList[I].DestFileName), false);

        // Notify that we downloaded a file
        Synchronize(UpdateProgressBar);

      end;

    except
      RaiseLastOSError;
    end;

  Except
    On e: exception do
    begin
      fErrOcur := true;
      fErrMsg := e.Message;
    end;
  end;
end;

Function tUpdateThread.IsUpdateNeeded: Boolean;
begin
  Result := False;
  try
    // ensure that the file is there
    fCriticalSec.Acquire;
    try
      fUpdateNeeded := false;

      // Clear the progressbar
      Synchronize(SetupProgCheck);

      // Build the File list
      BuildFileList;

      // Check if any need updates
      fUpdateNeeded := Length(fUpdatFileList) > 0;

    finally
      fCriticalSec.Release;
    end;

    Result := fUpdateNeeded;
  Except
    On e: exception do
    begin
      fErrOcur := true;
      fErrMsg := e.Message;
    end;
  end;
end;

function tUpdateThread.getFileDateTime(const FileName: string): TDateTime;
begin
   FileAge(FileName, Result);
end;
{$ENDREGION}

procedure TfrmUpdate.FormCreate(Sender: TObject);

  function GetAppDataPath: string;
  var
    Path: array [0 .. MAX_PATH] of Char;
  begin
    if SHGetFolderPath(0, CSIDL_LOCAL_APPDATA, 0, SHGFP_TYPE_CURRENT, Path) = S_OK
    then
      Result := IncludeTrailingPathDelimiter(Path)
    else
      Result := '';
  end;

begin
  fSourceFolder := GetCurrentDir;
  fDestFolder := IncludeTrailingPathDelimiter(GetAppDataPath) +
    'Mobile Electronic Documentation';
  fisWin7 := CheckWin32Version(6, 1);
  _TaskBarList := nil;
  _TaskbarList3 := nil;
  _TaskbarHandle := 0;

  //Setup Taskbar Progressbar
    if fisWin7 and fTaskProg then
    begin
      CoInitialize(nil);
      _TaskBarList := CreateComObject(CLSID_TaskbarList) as ITaskbarList;
      _TaskBarList.HrInit;
      Supports(_TaskBarList, IID_ITaskbarList3, _TaskbarList3);
      if Assigned(_TaskbarList3) then
      begin
        if Application.MainFormOnTaskBar and (Application.MainForm <> nil) then
          _TaskbarHandle := Application.MainForm.Handle
        else
          _TaskbarHandle := Application.Handle;
        try
          _TaskbarList3.SetProgressState(_TaskbarHandle, TBPF_NORMAL)
        except
          _TaskbarList3 := nil;
          _TaskBarList := nil;
          _TaskbarHandle := 0;
        end;
      end;
    end;

end;

function TfrmUpdate.GetDestFolder: String;
begin
  Result := fDestFolder;
end;

function TfrmUpdate.GetSourceFolder: String;
begin
  fSourceFolder := fSourceFolder;
end;

procedure TfrmUpdate.SetDestFolder(const Value: String);
begin
  fDestFolder := Value;
end;

procedure TfrmUpdate.SetSourceFolder(const Value: String);
begin
  fSourceFolder := Value;
end;

procedure TfrmUpdate.CreateParams(var Params: TCreateParams);
begin
  inherited;
  Params.ExStyle := Params.ExStyle or WS_EX_APPWINDOW;
end;

  procedure TfrmUpdate.StartFileTransfer();
  begin
    if not assigned(fTheThread) then
    begin
      // start the thread
      fTheThread := tUpdateThread.Create;
      fTheThread.SourceFolder := fSourceFolder;
      fTheThread.DestFolder := fDestFolder;
      fTheThread.HostForm := self;
      fTheThread.Start;
    end;
  end;

end.
