unit ORCtrlsVA508Compatibility;

interface

uses
  Forms, Controls, StdCtrls, SysUtils, Windows, VA508AccessibilityManager;

type
  TORCheckBox508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    function GetComponentName(Component: TWinControl): string; override;
    function GetInstructions(Component: TWinControl): string; override;
    function GetState(Component: TWinControl): string; override;
  end;

  TLBMgr = class
  private
    function GetIdx(Component: TWinControl): integer;
  public
    function GetComponentName(Component: TWinControl): string; virtual; abstract;
    function GetState(Component: TWinControl): string; virtual; abstract;
    function GetItemInstructions(Component: TWinControl): string; virtual; abstract;
  end;

  TORListBox508Manager = class(TVA508ManagedComponentClass)
  var
    FCheckBoxes: TLBMgr;
    FMultiSelect: TLBMgr;
    FStandard: TLBMgr;
    FCurrent: TLBMgr;
    function GetCurrent(Component: TWinControl): TLBMgr;
  public
    constructor Create; override;
    destructor Destroy; override;
    function GetComponentName(Component: TWinControl): string; override;
    function GetState(Component: TWinControl): string; override;
    function GetItem(Component: TWinControl): TObject; override;
    function GetItemInstructions(Component: TWinControl): string; override;
    function GetValue(Component: TWinControl): string; override;
  end;

  TVA508TORDateComboComplexManager = class(TVA508ComplexComponentManager)
  public
    constructor Create;
    procedure Refresh(Component: TWinControl;
                      AccessibilityManager: TVA508AccessibilityManager); override;
  end;


  TVA508TORCheckComboBoxComplexManager = class(TVA508ComplexComponentManager)
  public
    constructor Create;
    /// <summary>Ensure that the sub controls are registered with the
    /// AccessibilityManager. Called from
    /// <see cref="VA508AccessibilityManager|TVAGlobalComponentRegistry" /> in
    /// the 508 framework
    /// </summary>
    /// <param name="Component">
    ///   Main component registered to the 508 framework
    /// </param>
    /// <param name="AccessibilityManager">
    ///   508 manager used to govern these controls
    /// </param>
    procedure Refresh(Component: TWinControl;
                      AccessibilityManager: TVA508AccessibilityManager); override;
  end;

{  TVA508TORDateBoxComplexManager = class(TVA508ComplexComponentManager)
  public
    constructor Create;
    procedure Refresh(Component: TWinControl;
                      AccessibilityManager: TVA508AccessibilityManager); override;
  end;}

  TORComboBox508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    function GetValue(Component: TWinControl): string; override;
  end;

  TORCheckComboBox508Manager = class(TVA508ManagedComponentClass)
  public
   constructor Create; override;
    /// <summary>Called when getting the name of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and not using the default
    /// accessdata for this control in the 508 manager then the result will be
    /// the checkbox PLUS the checkbox caption </returns>
    function GetComponentName(Component: TWinControl): string; override;
    /// <summary>Called when getting the caption of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and main check box has focus
    /// the result will be the checkbox�s caption else it will be the
    /// component's caption</returns>
    function GetCaption(Component: TWinControl): string; override;
    /// <summary>Called when getting the instructions of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and main check box has focus
    /// the result will be the checkbox's instructions else it will be the
    /// component's instructions</returns>
    function GetInstructions(Component: TWinControl): string; override;
    /// <summary>Called when getting the state of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and main check box has focus
    /// the result will be the checkbox's state else it will be empty</returns>
    function GetState(Component: TWinControl): string; override;
    /// <summary>Called when getting the value of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and main check box has focus
    /// the result will be the empty else it will be the checkbox's
    /// text</returns>
    function GetValue(Component: TWinControl): string; override;
  end;

  TORDayCombo508Manager = class(TORComboBox508Manager)
  public
    constructor Create; override;
    function GetCaption(Component: TWinControl): string; override;
  end;

  TORMonthCombo508Manager = class(TORComboBox508Manager)
  public
    constructor Create; override;
    function GetCaption(Component: TWinControl): string; override;
  end;

  TORYearEdit508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    function GetCaption(Component: TWinControl): string; override;
  end;

  TORDateButton508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    function GetCaption(Component: TWinControl): string; override;
  end;


  TORComboEdit508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    /// <summary>Redirects the control class to another control's manager.
    /// Called from
    /// <see cref="VA508AccessibilityManager|InitializeComponentManager" /> in
    /// the 508 framework
    /// </summary>
    /// <param name="Component">
    ///   Main component to redirect in the 508 framework
    /// </param>
    /// <param name="ManagedType">
    ///   Managed type to handle during the redirect
    /// </param>
    /// <returns>The owner of Component as a TWinControl</returns>
    function Redirect(Component: TWinControl; var ManagedType: TManagedType): TWinControl; override;
  end;

  TORCheckComboCheckBoxMain508Manager = class(TVA508ManagedComponentClass)
  public
    constructor Create; override;
    /// <summary>Called when getting the caption of the component</summary>
    /// <param name="Component">Component used for the lookup</param>
    /// <returns>If assigned main checkbox and main check box has focus
    /// the result will be the checkbox�s caption else it will be the
    /// component's caption</returns>
    function GetComponentName(Component: TWinControl): string; override;
    /// <summary>Redirects the control class to another control's manager.
    /// Called from
    /// <see cref="VA508AccessibilityManager|InitializeComponentManager" /> in
    /// the 508 framework
    /// </summary>
    /// <param name="Component">
    ///   Main component to redirect in the 508 framework
    /// </param>
    /// <param name="ManagedType">
    ///   Managed type to handle during the redirect
    /// </param>
    /// <returns>The owner of Component as a TWinControl</returns>
    function Redirect(Component: TWinControl; var ManagedType: TManagedType): TWinControl; override;
  end;



//  TORComboCheckBoxMain508Manager = class(TVA508ManagedComponentClass)
//  public
//    constructor Create; override;
//    function GetComponentName(Component: TWinControl): string; override;
//   function Redirect(Component: TWinControl; var ManagedType: TManagedType): TWinControl; override;
//  end;


implementation

uses VA508DelphiCompatibility, ORCtrls, ORDtTm, VA508AccessibilityRouter,
  VA508AccessibilityConst, ORDtTmRng, ORCheckComboBox, UResponsiveGUI;

/// <summary>Search for a given sub control based on control type</summary>
/// <param name="Component">Base component used in the search</param>
/// <param name="FindControlClass">Control class to search for</param>
/// <returns>The sub control (if found) or nil (if not found)</returns>
function FindControlType(Component: TWinControl; FindControlClass: TClass)
  : TWinControl;
var
  i: integer;
  control: TWinControl;
begin
  Result := nil;

  If Component is FindControlClass then
    Result := Component
  else
  begin
    for i := 0 to Component.ControlCount - 1 do
    begin
      if Assigned(Result) then
        break;
      if Component.Controls[i] is TWinControl then
      begin
        control := TWinControl(Component.Controls[i]);
        Result := FindControlType(control, FindControlClass);
      end;
    end;
  end;
end;

/// <summary>Find the edit box for a TORComboBox</summary>
/// <param name="ComboBox">TORComboBox used in the search</param>
/// <returns>The TORComboEdit control (if found) or nil (if not found)</returns>
function GetEditBox(ComboBox: TORComboBox): TORComboEdit;
begin
  Result := TORComboEdit(FindControlType(ComboBox, TORComboEdit));
end;

/// <summary>Find the check box for a TORComboBox</summary>
/// <param name="ComboBox">TORComboBox used in the search</param>
/// <returns>The TCheckBox control (if found) or nil (if not found)</returns>
function GetCheckBox(ComboBox: TORComboBox): TCheckBox;
begin
  Result := TCheckBox(FindControlType(ComboBox, TCheckBox));
end;

//currently not used -- see below
function ORComboBoxAlternateHandle(Component: TWinControl): HWnd;
var
  eBox: TORComboEdit;
  cBox: TORComboBox;
begin
  cBox := TORComboBox(Component);
  eBox := GetEditBox(cBox);

  if Assigned(eBox) then
    Result := eBox.Handle
  else
    Result := cBox.Handle;
end;


type
  TVA508RegistrationScreenReader = class(TVA508ScreenReader);
{ Registration }

procedure RegisterORComponents;
begin
  RegisterAlternateHandleComponent(TORComboBox, ORComboBoxAlternateHandle, [TORCheckComboBox]);
  RegisterManagedComponentClass(TORCheckBox508Manager.Create);
 // RegisterManagedComponentClass(TORComboBox508Manager.Create);
  RegisterManagedComponentClass(TORListBox508Manager.Create);
  RegisterManagedComponentClass(TORDayCombo508Manager.Create);
  RegisterManagedComponentClass(TORMonthCombo508Manager.Create);
  RegisterManagedComponentClass(TORYearEdit508Manager.Create);
  RegisterManagedComponentClass(TORDateButton508Manager.Create);
  RegisterManagedComponentClass(TORComboEdit508Manager.Create);

  RegisterManagedComponentClass(TORCheckComboBox508Manager.Create);
  RegisterManagedComponentClass(TORCheckComboCheckBoxMain508Manager.Create);
  RegisterComplexComponentManager(TVA508TORCheckComboBoxComplexManager.Create);

  RegisterComplexComponentManager(TVA508TORDateComboComplexManager.Create);


 // RegisterComplexComponentManager(TVA508TORDateBoxComplexManager.Create);



  with TVA508RegistrationScreenReader(GetScreenReader) do
  begin
//---TORCalendar ???
//---TORPopupMenu ???
//---TORMenuItem ???

    RegisterCustomClassBehavior(TORTreeView.ClassName, CLASS_BEHAVIOR_TREE_VIEW);
    RegisterCustomClassBehavior(TORAlignEdit.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TORAlignButton.ClassName, CLASS_BEHAVIOR_BUTTON);
    RegisterCustomClassBehavior(TORAlignSpeedButton.ClassName, CLASS_BEHAVIOR_BUTTON);
    RegisterCustomClassBehavior(TORCheckBox.ClassName, CLASS_BEHAVIOR_CHECK_BOX);
    RegisterCustomClassBehavior(TKeyClickPanel.ClassName, CLASS_BEHAVIOR_BUTTON);
    RegisterCustomClassBehavior(TKeyClickRadioGroup.ClassName, CLASS_BEHAVIOR_GROUP_BOX);
    RegisterCustomClassBehavior(TCaptionTreeView.ClassName, CLASS_BEHAVIOR_TREE_VIEW);
    RegisterCustomClassBehavior(TCaptionMemo.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TCaptionEdit.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TCaptionRichEdit.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TOROffsetLabel.ClassName, CLASS_BEHAVIOR_STATIC_TEXT);

    RegisterCustomClassBehavior(TCaptionComboBox.ClassName, CLASS_BEHAVIOR_COMBO_BOX);
    RegisterCustomClassBehavior(TORComboEdit.ClassName, CLASS_BEHAVIOR_EDIT_COMBO);
    RegisterCustomClassBehavior(TORComboBox.ClassName, CLASS_BEHAVIOR_COMBO_BOX);
    RegisterCustomClassBehavior(TORListBox.ClassName, CLASS_BEHAVIOR_LIST_BOX);
    RegisterCustomClassBehavior(TCaptionCheckListBox.ClassName, CLASS_BEHAVIOR_LIST_BOX);
    RegisterCustomClassBehavior(TCaptionStringGrid.ClassName, CLASS_BEHAVIOR_LIST_BOX);

    RegisterCustomClassBehavior(TORDateEdit.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TORDayCombo.ClassName, CLASS_BEHAVIOR_COMBO_BOX);
    RegisterCustomClassBehavior(TORMonthCombo.ClassName, CLASS_BEHAVIOR_COMBO_BOX);
    RegisterCustomClassBehavior(TORYearEdit.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TORDateBox.ClassName, CLASS_BEHAVIOR_EDIT);
    RegisterCustomClassBehavior(TORDateCombo.ClassName, CLASS_BEHAVIOR_GROUP_BOX);

    RegisterCustomClassBehavior(TORListView.ClassName, CLASS_BEHAVIOR_LIST_VIEW);
    RegisterCustomClassBehavior(TCaptionListView.ClassName, CLASS_BEHAVIOR_LIST_VIEW);
    RegisterCustomClassBehavior(TCaptionListBox.ClassName, CLASS_BEHAVIOR_LIST_BOX);

    RegisterCustomClassBehavior(TORDateRangeDlg.ClassName, CLASS_BEHAVIOR_DIALOG);
    RegisterCustomClassBehavior(TORfrmDtTm.ClassName, CLASS_BEHAVIOR_DIALOG);//called by TORDateTimeDlg
  end;
end;

{ TORCheckBox508Manager }

constructor TORCheckBox508Manager.Create;
begin
  inherited Create(TORCheckBox, [mtComponentName, mtInstructions, mtState, mtStateChange], TRUE);
end;

function TORCheckBox508Manager.GetComponentName(Component: TWinControl): string;
begin
  with TORCheckBox(Component) do
  begin
    if RadioStyle then
      Result := 'radio button'
    else
      Result := VA508DelphiCompatibility.GetCheckBoxComponentName(AllowGrayed);
  end;
end;

function TORCheckBox508Manager.GetInstructions(Component: TWinControl): string;
begin
  Result := VA508DelphiCompatibility.GetCheckBoxInstructionMessage(TORCheckBox(Component).Checked);
end;

function TORCheckBox508Manager.GetState(Component: TWinControl): string;
var
  cb: TORCheckBox;
begin
  TResponsiveGUI.ProcessMessages; // <<<  needed to allow messages that set state to process
  Result := '';
  cb := TORCheckBox(Component);
  if (cb.State = cbGrayed) and (cb.GrayedStyle in [gsQuestionMark, gsBlueQuestionMark]) then
    Result := 'Question Mark'
  else
    Result := VA508DelphiCompatibility.GetCheckBoxStateText(cb.State);
end;

{ TORListBox508Manager }

type
  TORListBoxCheckBoxes508Manager = class(TLBMgr)
  public
    function GetComponentName(Component: TWinControl): string; override;
    function GetState(Component: TWinControl): string; override;
    function GetItemInstructions(Component: TWinControl): string; override;
  end;

  TORListBoxMultiSelect508Manager = class(TLBMgr)
  public
    function GetComponentName(Component: TWinControl): string; override;
    function GetState(Component: TWinControl): string; override;
    function GetItemInstructions(Component: TWinControl): string; override;
  end;

  TORListBoxStandard508Manager = class(TLBMgr)
  public
    function GetComponentName(Component: TWinControl): string; override;
    function GetState(Component: TWinControl): string; override;
    function GetItemInstructions(Component: TWinControl): string; override;
  end;

constructor TORListBox508Manager.Create;
begin
  inherited Create(TORListBox, [mtComponentName, mtValue, mtState, mtStateChange,
                   mtItemChange, mtItemInstructions]);
end;

destructor TORListBox508Manager.Destroy;
begin
  FCurrent := nil;
  if assigned(FCheckBoxes) then
    FreeAndNil(FCheckBoxes);
  if assigned(FMultiSelect) then
    FreeAndNil(FMultiSelect);
  if assigned(FStandard) then
    FreeAndNil(FStandard);
  inherited;
end;

function TORListBox508Manager.GetComponentName(Component: TWinControl): string;
begin
  Result := GetCurrent(Component).GetComponentName(Component);
end;

function TORListBox508Manager.GetItem(Component: TWinControl): TObject;
var
  lb : TORListBox;
  max, id: integer;
begin
  GetCurrent(Component);
  lb := TORListBox(Component);
  max := lb.items.Count + 2;
  if max < 10000 then
    max := 10000;
  id := (lb.items.Count * max) + (lb.FocusIndex + 2);
  if lb.FocusIndex < 0 then dec(id);
  Result := TObject(id);
end;

function TORListBox508Manager.GetItemInstructions(
  Component: TWinControl): string;
begin
  Result := GetCurrent(Component).GetItemInstructions(Component);
end;

function TORListBox508Manager.GetState(Component: TWinControl): string;
begin
  Result := GetCurrent(Component).GetState(Component);
end;

function TORListBox508Manager.GetValue(Component: TWinControl): string;
var idx: integer;
  lb: TORListBox;
begin
  lb := TORListBox(Component);
  idx := lb.FocusIndex;
  if idx < 0 then
    idx := 0;
  Result := lb.DisplayText[idx];
end;

function TORListBox508Manager.GetCurrent(Component: TWinControl): TLBMgr;
var
  lb : TORListBox;

begin
  lb := TORListBox(Component);
  if lb.CheckBoxes then
  begin
    if not assigned(FCheckBoxes) then
      FCheckBoxes := TORListBoxCheckBoxes508Manager.Create;
    FCurrent := FCheckBoxes;
  end
  else if lb.MultiSelect then
  begin
    if not assigned(FMultiSelect) then
      FMultiSelect := TORListBoxMultiSelect508Manager.Create;
    FCurrent := FMultiSelect;
  end
  else
  begin
    if not assigned(FStandard) then
      FStandard := TORListBoxStandard508Manager.Create;
    FCurrent := FStandard;
  end;
  Result := FCurrent;
end;

{ TORListBoxCheckBoxes508Manager }

function TORListBoxCheckBoxes508Manager.GetComponentName(
  Component: TWinControl): string;
begin
  Result := 'Check List Box'
end;

function TORListBoxCheckBoxes508Manager.GetItemInstructions(
  Component: TWinControl): string;
var
  lb: TORListBox;
  idx: integer;
begin
  Result := '';
  lb := TORListBox(Component);
  idx := GetIdx(Component);
  if (idx >= 0) then
    Result := VA508DelphiCompatibility.GetCheckBoxInstructionMessage(lb.Checked[idx])
  else
    Result := '';
end;

function TORListBoxCheckBoxes508Manager.GetState(
  Component: TWinControl): string;
var
  lb: TORListBox;
  idx: integer;
begin
  lb := TORListBox(Component);
  idx := GetIdx(Component);
  if (idx >= 0) then
  begin
    Result := GetCheckBoxStateText(lb.CheckedState[idx]);
    if lb.FocusIndex < 0 then
      Result := 'not selected ' + Result;
  end
  else
    Result := '';
end;

{ TORListBoxMultiSelect508Manager }

function TORListBoxMultiSelect508Manager.GetComponentName(
  Component: TWinControl): string;
begin
  Result := 'Multi Select List Box'
end;

function TORListBoxMultiSelect508Manager.GetItemInstructions(
  Component: TWinControl): string;
var
  lb: TORListBox;
  idx: integer;
begin
  Result := '';
  lb := TORListBox(Component);
  idx := GetIdx(Component);
  if (idx >= 0) then
  begin
    if not lb.Selected[idx] then
      Result := 'to select press space bar'
    else
      Result := 'to un select press space bar';
  end;
end;

function TORListBoxMultiSelect508Manager.GetState(
  Component: TWinControl): string;
var
  lb: TORListBox;
  idx: Integer;
begin
  lb := TORListBox(Component);
  idx := GetIdx(Component);
  if (idx >= 0) then
  begin
    if lb.Selected[idx] then
      Result := 'Selected'
    else
      Result := 'Not Selected';
  end
  else
    Result := '';
end;

{ TORListBoxStandard508Manager }

function TORListBoxStandard508Manager.GetComponentName(
  Component: TWinControl): string;
begin
  Result := 'List Box';
end;

function TORListBoxStandard508Manager.GetItemInstructions(
  Component: TWinControl): string;
begin
  Result := '';
end;

function TORListBoxStandard508Manager.GetState(Component: TWinControl): string;
var
  lb: TORListBox;
begin
  lb := TORListBox(Component);
  if (lb.FocusIndex < 0) then
    Result := 'Not Selected'
  else
    Result := '';
end;

{ TLBMgr }

function TLBMgr.GetIdx(Component: TWinControl): integer;
begin
  Result := TORListBox(Component).FocusIndex;
  if (Result < 0) and (TORListBox(Component).Count > 0) then
    Result := 0;
end;

{ TVA508TORDateComboComplexManager }

constructor TVA508TORDateComboComplexManager.Create;
begin
  inherited Create(TORDateCombo);
end;

type
  TORDateComboFriend = class(TORDateCombo);

procedure TVA508TORDateComboComplexManager.Refresh(Component: TWinControl;
  AccessibilityManager: TVA508AccessibilityManager);
begin
  with TORDateComboFriend(Component) do
  begin
    ClearSubControls(Component);
//    if assigned(CalBtn) then
//      CalBtn.TabStop := TRUE;
//    if IncludeBtn then
//      AddSubControl(CalBtn, AccessibilityManager);
    AddSubControl(Component, YearEdit, AccessibilityManager);
//    AddSubControl(YearUD, AccessibilityManager);
    if IncludeMonth then
      AddSubControl(Component, MonthCombo, AccessibilityManager);
    if IncludeDay then
      AddSubControl(Component, DayCombo, AccessibilityManager);
  end;
end;

{ TORDayCombo508Manager }

constructor TORDayCombo508Manager.Create;
begin
  inherited Create(TORDayCombo, [mtCaption, mtValue]);
end;

function TORDayCombo508Manager.GetCaption(Component: TWinControl): string;
begin
  Result := 'Day';
end;

{ TORMonthCombo508Manager }

constructor TORMonthCombo508Manager.Create;
begin
  inherited Create(TORMonthCombo, [mtCaption, mtValue]);
end;

function TORMonthCombo508Manager.GetCaption(Component: TWinControl): string;
begin
  Result := 'Month';
end;

{ TORYearEdit508Manager }

constructor TORYearEdit508Manager.Create;
begin
  inherited Create(TORYearEdit, [mtCaption]);
end;

function TORYearEdit508Manager.GetCaption(Component: TWinControl): string;
begin
  Result := 'Year';
end;

{ TORDateButton508Manager }

constructor TORDateButton508Manager.Create;
begin
  inherited Create(TORDateButton, [mtCaption]);
end;

function TORDateButton508Manager.GetCaption(Component: TWinControl): string;
begin
  Result := 'Date';
end;

(*
{ TVA508TORDateBoxComplexManager }

constructor TVA508TORDateBoxComplexManager.Create;
begin
  inherited Create(TORDateBox);
end;

type
  TORDateBoxFriend = class(TORDateBox);

procedure TVA508TORDateBoxComplexManager.Refresh(Component: TWinControl;
  AccessibilityManager: TVA508AccessibilityManager);
begin
  with TORDateBoxFriend(Component) do
  begin
    ClearSubControls;
    if assigned(DateButton) then
    begin
      DateButton.TabStop := TRUE;
      AddSubControl(DateButton, AccessibilityManager);
    end;
  end;
end;
*)

{ TVA508ORComboManager }

constructor TORComboBox508Manager.Create;
begin
  inherited Create(TORComboBox, [mtValue], True);
end;

function TORComboBox508Manager.GetValue(Component: TWinControl): string;
begin
  Result := TORComboBox(Component).Text;
end;

{ TORCheckComboBox508Manager }

constructor TORCheckComboBox508Manager.Create;
begin
  inherited Create(TORCheckComboBox, [mtValue, mtComponentName, mtInstructions, mtState, mtStateChange, mtCaption], TRUE);
end;

function TORCheckComboBox508Manager.GetValue(Component: TWinControl): string;
var
  eBox: TORComboEdit;
  eCheckBox: TCheckBox;
begin
  eCheckBox := GetCheckBox(TORComboBox(Component));
  eBox := GetEditBox(TORComboBox(Component));

  If assigned(eCheckBox) and eCheckBox.Focused then
    Result := ''
  else if assigned(eBox) then
    Result := TORComboBox(Component).Text;
end;

function TORCheckComboBox508Manager.GetCaption(Component: TWinControl): string;
var
  eBox: TORComboEdit;
  eCheckBox: TORCheckBox;
begin
  eCheckBox := TORCheckBox(GetCheckBox(TORComboBox(Component)));
  eBox := GetEditBox(TORComboBox(Component));
  If assigned(eCheckBox) and eCheckBox.Focused then
  begin
     Result := eCheckBox.Caption
  end else if assigned(eBox) then
    Result := TORComboBox(Component).caption;
end;

function TORCheckComboBox508Manager.GetComponentName(Component: TWinControl): string;
var
  eBox: TORComboEdit;
  eCheckBox: TORCheckBox;
begin
  eCheckBox := TORCheckBox(GetCheckBox(TORComboBox(Component)));
  eBox := GetEditBox(TORComboBox(Component));
  If assigned(eCheckBox) and eCheckBox.Focused then
  begin
   if eCheckBox.RadioStyle then
      Result := 'radio button'
    else
      Result := VA508DelphiCompatibility.GetCheckBoxComponentName(eCheckBox.AllowGrayed);

    Result := Result + ', ' + eCheckBox.Caption;

  end else if assigned(eBox) then
    Result := 'Edit Box';
end;

function TORCheckComboBox508Manager.GetInstructions(Component: TWinControl): string;
var
  eCheckBox: TORCheckBox;
begin
  eCheckBox := TORCheckBox(GetCheckBox(TORComboBox(Component)));
  If assigned(eCheckBox) and eCheckBox.Focused then
    Result := VA508DelphiCompatibility.GetCheckBoxInstructionMessage(eCheckBox.Checked)
  else
   Result := Inherited GetInstructions(Component);
end;

function TORCheckComboBox508Manager.GetState(Component: TWinControl): string;
var
  eCheckBox: TORCheckBox;
begin
  Application.ProcessMessages;
  eCheckBox := TORCheckBox(GetCheckBox(TORComboBox(Component)));
  If assigned(eCheckBox) and eCheckBox.Focused then
  begin
    if (eCheckBox.State = cbGrayed) and (eCheckBox.GrayedStyle in [gsQuestionMark, gsBlueQuestionMark]) then
      Result := 'Question Mark'
    else
       Result := VA508DelphiCompatibility.GetCheckBoxStateText(eCheckBox.State);
  end else
    Result := '';
end;


{ TORComboEdit508Manager }

constructor TORComboEdit508Manager.Create;
begin
  inherited Create(TORComboEdit, [mtComponentRedirect]);
end;
//end;

//constructor TORComboCheckBoxMain508Manager.Create;
//begin
//  inherited Create(TORComboCheckBoxMain, [mtComponentRedirect, mtComponentName, mtInstructions, mtState, mtStateChange]);
//end;
//
//function TORComboCheckBoxMain508Manager.Redirect(Component: TWinControl;
//  var ManagedType: TManagedType): TWinControl;
//begin
//  ManagedType := mtCaption;
//  Result := (Component.Owner as TWinControl);
//end;
//
//function TORComboCheckBoxMain508Manager.GetComponentName
//  (Component: TWinControl): string;
//begin
//  Result := '';
//  if Assigned(Component) then
//  begin
//    with (Component as TORCheckBox) do
//    begin
//      if RadioStyle then
//        Result := 'radio button'
//      else
//        Result := VA508DelphiCompatibility.GetCheckBoxComponentName
//          (AllowGrayed);
//    end;
//  end;

function TORComboEdit508Manager.Redirect(Component: TWinControl;
  var ManagedType: TManagedType): TWinControl;
begin
  ManagedType := mtCaption;
  Result := (Component.Owner as TWinControl);
end;


{ TORCheckComboCheckBoxMain508Manager }

constructor TORCheckComboCheckBoxMain508Manager.Create;
begin
  inherited Create(TORCheckComboCheckBoxMain, [mtComponentRedirect, mtComponentName, mtInstructions, mtState, mtStateChange]);
end;

function TORCheckComboCheckBoxMain508Manager.Redirect(Component: TWinControl;
  var ManagedType: TManagedType): TWinControl;
begin
  ManagedType := mtCaption;
  Result := (Component.Owner as TWinControl);
end;

function TORCheckComboCheckBoxMain508Manager.GetComponentName
  (Component: TWinControl): string;
begin
  Result := '';
  if Assigned(Component) then
  begin
    with (Component as TORCheckBox) do
    begin
      if RadioStyle then
        Result := 'radio button'
      else
        Result := VA508DelphiCompatibility.GetCheckBoxComponentName
          (AllowGrayed);
    end;
  end;
end;

{ TVA508TORCheckComboBoxComplexManager }

constructor TVA508TORCheckComboBoxComplexManager.Create;
begin
  inherited Create(TORCheckComboBox);
end;

procedure TVA508TORCheckComboBoxComplexManager.Refresh(Component: TWinControl;
  AccessibilityManager: TVA508AccessibilityManager);
var
  eBox: TORComboEdit;
  eCheckBox: TCheckBox;
begin
  ClearSubControls(Component);
  eBox := GetEditBox(TORCheckComboBox(Component));
  if Assigned(eBox) then
    AddSubControl(Component, eBox, AccessibilityManager);

  eCheckBox := GetCheckBox(TORCheckComboBox(Component));
  if Assigned(eCheckBox) then
    AddSubControl(Component, eCheckBox, AccessibilityManager);
end;

initialization
  RegisterORComponents;

end.
