unit UWinMsgNames;

interface

uses
  System.Classes;

type
  TWinMsgNames = class(TObject)
  public const
    Delim = '/';
  private type
    TMessageName = class(TObject)
    strict private
      FMessageNames: TStringList;
    public
      constructor Create(AName: string);
      destructor Destroy; override;
      procedure Add(AName: string);
      function ToString: string; override;
    end;
  private
    class var FNames: TStringList;
    class constructor Create;
    class destructor Destroy;
    class function GetName(Msg: Cardinal): string; static;
    class procedure SetName(Msg: Cardinal; const Value: string); static;
    class function MessageToSortableID(Msg: Cardinal): string;
    class procedure InitDefaultList;
  public
    /// <summary>The windows message name of the passed in value.</summary>
    /// <param name="Msg">[<see cref="system|Cardinal"/>]The message number.</param>
    /// <returns>[<see cref="system|string"/>]- The name of the window's message,
    ///    or, if there is no name found, the hexidecimal value of the message
    ///    number.  If multiple message names share the same message number, they
    ///    are displayed with separating "/" characters.</returns>
    class property Name[Msg: Cardinal]: string read GetName write SetName;
  end;

implementation

uses
  System.SysUtils,
  Vcl.Controls,
  Winapi.CommCtrl,
  Winapi.Messages,
  Winapi.RichEdit;

{ TWinMsgNames.TMessageName }

procedure TWinMsgNames.TMessageName.Add(AName: string);
begin
  FMessageNames.Add(AName);
end;

constructor TWinMsgNames.TMessageName.Create(AName: string);
begin
  inherited Create;
  FMessageNames := TStringList.Create(dupIgnore, True, False);
  FMessageNames.Add(AName);
end;

destructor TWinMsgNames.TMessageName.Destroy;
begin
  FreeAndNil(FMessageNames);
  inherited;
end;

// Returns sorted list of message names for the given code
function TWinMsgNames.TMessageName.ToString: string;
var
  i: Integer;
begin
  Result := '';
  for i := 0 to FMessageNames.Count - 1 do
  begin
    if i > 0 then
      Result := Result + TWinMsgNames.Delim;
    Result := Result + FMessageNames[i];
  end;
end;

{ TWinMsgNames }

class constructor TWinMsgNames.Create;
begin
  FNames := TStringList.Create(True);
  FNames.CaseSensitive := True;
  FNames.Sorted := True;
  InitDefaultList;
end;

class destructor TWinMsgNames.Destroy;
begin
  FreeAndNil(FNames);
end;

class function TWinMsgNames.GetName(Msg: Cardinal): string;
var
  Index: Integer;
begin
  if FNames.Find(MessageToSortableID(Msg), Index) then
    Result := (FNames.Objects[Index] as TMessageName).ToString
  else
    Result := Format('$%x', [Msg]);
end;

class procedure TWinMsgNames.SetName(Msg: Cardinal; const Value: string);
var
  Index: Integer;
  MsgID: string;
  MsgName: TMessageName;
begin
  MsgID := MessageToSortableID(Msg);
  if FNames.Find(MsgID, Index) then
  begin
    MsgName := FNames.Objects[Index] as TMessageName;
    MsgName.Add(Value);
  end
  else
    FNames.AddObject(MsgID, TMessageName.Create(Value));
end;

class function TWinMsgNames.MessageToSortableID(Msg: Cardinal): string;
begin
  Result := Format('%8.8x', [Msg]);
end;

class procedure TWinMsgNames.InitDefaultList;
begin
  Name[WM_NULL] := 'WM_NULL';
  Name[WM_CREATE] := 'WM_CREATE';
  Name[WM_DESTROY] := 'WM_DESTROY';
  Name[WM_MOVE] := 'WM_MOVE';
  Name[WM_SIZE] := 'WM_SIZE';
  Name[WM_ACTIVATE] := 'WM_ACTIVATE';
  Name[WM_SETFOCUS] := 'WM_SETFOCUS';
  Name[WM_KILLFOCUS] := 'WM_KILLFOCUS';
  Name[WM_ENABLE] := 'WM_ENABLE';
  Name[WM_SETREDRAW] := 'WM_SETREDRAW';
  Name[WM_SETTEXT] := 'WM_SETTEXT';
  Name[WM_GETTEXT] := 'WM_GETTEXT';
  Name[WM_GETTEXTLENGTH] := 'WM_GETTEXTLENGTH';
  Name[WM_PAINT] := 'WM_PAINT';
  Name[WM_CLOSE] := 'WM_CLOSE';
  Name[WM_QUERYENDSESSION] := 'WM_QUERYENDSESSION';
  Name[WM_QUIT] := 'WM_QUIT';
  Name[WM_QUERYOPEN] := 'WM_QUERYOPEN';
  Name[WM_ERASEBKGND] := 'WM_ERASEBKGND';
  Name[WM_SYSCOLORCHANGE] := 'WM_SYSCOLORCHANGE';
  Name[WM_ENDSESSION] := 'WM_ENDSESSION';
  Name[WM_SHOWWINDOW] := 'WM_SHOWWINDOW';
  Name[WM_WININICHANGE] := 'WM_WININICHANGE';
  Name[WM_SETTINGCHANGE] := 'WM_SETTINGCHANGE';
  Name[WM_DEVMODECHANGE] := 'WM_DEVMODECHANGE';
  Name[WM_ACTIVATEAPP] := 'WM_ACTIVATEAPP';
  Name[WM_FONTCHANGE] := 'WM_FONTCHANGE';
  Name[WM_TIMECHANGE] := 'WM_TIMECHANGE';
  Name[WM_CANCELMODE] := 'WM_CANCELMODE';
  Name[WM_SETCURSOR] := 'WM_SETCURSOR';
  Name[WM_MOUSEACTIVATE] := 'WM_MOUSEACTIVATE';
  Name[WM_CHILDACTIVATE] := 'WM_CHILDACTIVATE';
  Name[WM_QUEUESYNC] := 'WM_QUEUESYNC';
  Name[WM_GETMINMAXINFO] := 'WM_GETMINMAXINFO';
  Name[WM_PAINTICON] := 'WM_PAINTICON';
  Name[WM_ICONERASEBKGND] := 'WM_ICONERASEBKGND';
  Name[WM_NEXTDLGCTL] := 'WM_NEXTDLGCTL';
  Name[WM_SPOOLERSTATUS] := 'WM_SPOOLERSTATUS';
  Name[WM_DRAWITEM] := 'WM_DRAWITEM';
  Name[WM_MEASUREITEM] := 'WM_MEASUREITEM';
  Name[WM_DELETEITEM] := 'WM_DELETEITEM';
  Name[WM_VKEYTOITEM] := 'WM_VKEYTOITEM';
  Name[WM_CHARTOITEM] := 'WM_CHARTOITEM';
  Name[WM_SETFONT] := 'WM_SETFONT';
  Name[WM_GETFONT] := 'WM_GETFONT';
  Name[WM_SETHOTKEY] := 'WM_SETHOTKEY';
  Name[WM_GETHOTKEY] := 'WM_GETHOTKEY';
  Name[WM_QUERYDRAGICON] := 'WM_QUERYDRAGICON';
  Name[WM_COMPAREITEM] := 'WM_COMPAREITEM';
  Name[WM_GETOBJECT] := 'WM_GETOBJECT';
  Name[WM_COMPACTING] := 'WM_COMPACTING';
  Name[WM_COMMNOTIFY] := 'WM_COMMNOTIFY';
  Name[WM_WINDOWPOSCHANGING] := 'WM_WINDOWPOSCHANGING';
  Name[WM_WINDOWPOSCHANGED] := 'WM_WINDOWPOSCHANGED';
  Name[WM_POWER] := 'WM_POWER';
  Name[WM_COPYDATA] := 'WM_COPYDATA';
  Name[WM_CANCELJOURNAL] := 'WM_CANCELJOURNAL';
  Name[WM_NOTIFY] := 'WM_NOTIFY';
  Name[WM_INPUTLANGCHANGEREQUEST] := 'WM_INPUTLANGCHANGEREQUEST';
  Name[WM_INPUTLANGCHANGE] := 'WM_INPUTLANGCHANGE';
  Name[WM_TCARD] := 'WM_TCARD';
  Name[WM_HELP] := 'WM_HELP';
  Name[WM_USERCHANGED] := 'WM_USERCHANGED';
  Name[WM_NOTIFYFORMAT] := 'WM_NOTIFYFORMAT';
  Name[WM_CONTEXTMENU] := 'WM_CONTEXTMENU';
  Name[WM_STYLECHANGING] := 'WM_STYLECHANGING';
  Name[WM_STYLECHANGED] := 'WM_STYLECHANGED';
  Name[WM_DISPLAYCHANGE] := 'WM_DISPLAYCHANGE';
  Name[WM_GETICON] := 'WM_GETICON';
  Name[WM_SETICON] := 'WM_SETICON';
  Name[WM_NCCREATE] := 'WM_NCCREATE';
  Name[WM_NCDESTROY] := 'WM_NCDESTROY';
  Name[WM_NCCALCSIZE] := 'WM_NCCALCSIZE';
  Name[WM_NCHITTEST] := 'WM_NCHITTEST';
  Name[WM_NCPAINT] := 'WM_NCPAINT';
  Name[WM_NCACTIVATE] := 'WM_NCACTIVATE';
  Name[WM_GETDLGCODE] := 'WM_GETDLGCODE';
  Name[WM_NCMOUSEMOVE] := 'WM_NCMOUSEMOVE';
  Name[WM_NCLBUTTONDOWN] := 'WM_NCLBUTTONDOWN';
  Name[WM_NCLBUTTONUP] := 'WM_NCLBUTTONUP';
  Name[WM_NCLBUTTONDBLCLK] := 'WM_NCLBUTTONDBLCLK';
  Name[WM_NCRBUTTONDOWN] := 'WM_NCRBUTTONDOWN';
  Name[WM_NCRBUTTONUP] := 'WM_NCRBUTTONUP';
  Name[WM_NCRBUTTONDBLCLK] := 'WM_NCRBUTTONDBLCLK';
  Name[WM_NCMBUTTONDOWN] := 'WM_NCMBUTTONDOWN';
  Name[WM_NCMBUTTONUP] := 'WM_NCMBUTTONUP';
  Name[WM_NCMBUTTONDBLCLK] := 'WM_NCMBUTTONDBLCLK';
  Name[WM_NCXBUTTONDOWN] := 'WM_NCXBUTTONDOWN';
  Name[WM_NCXBUTTONUP] := 'WM_NCXBUTTONUP';
  Name[WM_NCXBUTTONDBLCLK] := 'WM_NCXBUTTONDBLCLK';
  Name[WM_INPUT_DEVICE_CHANGE] := 'WM_INPUT_DEVICE_CHANGE';
  Name[WM_INPUT] := 'WM_INPUT';
  Name[WM_KEYDOWN] := 'WM_KEYDOWN';
  Name[WM_KEYUP] := 'WM_KEYUP';
  Name[WM_CHAR] := 'WM_CHAR';
  Name[WM_DEADCHAR] := 'WM_DEADCHAR';
  Name[WM_SYSKEYDOWN] := 'WM_SYSKEYDOWN';
  Name[WM_SYSKEYUP] := 'WM_SYSKEYUP';
  Name[WM_SYSCHAR] := 'WM_SYSCHAR';
  Name[WM_SYSDEADCHAR] := 'WM_SYSDEADCHAR';
  Name[WM_UNICHAR] := 'WM_UNICHAR';
  Name[WM_INITDIALOG] := 'WM_INITDIALOG';
  Name[WM_COMMAND] := 'WM_COMMAND';
  Name[WM_SYSCOMMAND] := 'WM_SYSCOMMAND';
  Name[WM_TIMER] := 'WM_TIMER';
  Name[WM_HSCROLL] := 'WM_HSCROLL';
  Name[WM_VSCROLL] := 'WM_VSCROLL';
  Name[WM_INITMENU] := 'WM_INITMENU';
  Name[WM_INITMENUPOPUP] := 'WM_INITMENUPOPUP';
  Name[WM_GESTURE] := 'WM_GESTURE';
  Name[WM_GESTURENOTIFY] := 'WM_GESTURENOTIFY';
  Name[WM_MENUSELECT] := 'WM_MENUSELECT';
  Name[WM_MENUCHAR] := 'WM_MENUCHAR';
  Name[WM_ENTERIDLE] := 'WM_ENTERIDLE';
  Name[WM_MENURBUTTONUP] := 'WM_MENURBUTTONUP';
  Name[WM_MENUDRAG] := 'WM_MENUDRAG';
  Name[WM_MENUGETOBJECT] := 'WM_MENUGETOBJECT';
  Name[WM_UNINITMENUPOPUP] := 'WM_UNINITMENUPOPUP';
  Name[WM_MENUCOMMAND] := 'WM_MENUCOMMAND';
  Name[WM_CHANGEUISTATE] := 'WM_CHANGEUISTATE';
  Name[WM_UPDATEUISTATE] := 'WM_UPDATEUISTATE';
  Name[WM_QUERYUISTATE] := 'WM_QUERYUISTATE';
  Name[WM_CTLCOLORMSGBOX] := 'WM_CTLCOLORMSGBOX';
  Name[WM_CTLCOLOREDIT] := 'WM_CTLCOLOREDIT';
  Name[WM_CTLCOLORLISTBOX] := 'WM_CTLCOLORLISTBOX';
  Name[WM_CTLCOLORBTN] := 'WM_CTLCOLORBTN';
  Name[WM_CTLCOLORDLG] := 'WM_CTLCOLORDLG';
  Name[WM_CTLCOLORSCROLLBAR] := 'WM_CTLCOLORSCROLLBAR';
  Name[WM_CTLCOLORSTATIC] := 'WM_CTLCOLORSTATIC';
  Name[WM_MOUSEMOVE] := 'WM_MOUSEMOVE';
  Name[WM_LBUTTONDOWN] := 'WM_LBUTTONDOWN';
  Name[WM_LBUTTONUP] := 'WM_LBUTTONUP';
  Name[WM_LBUTTONDBLCLK] := 'WM_LBUTTONDBLCLK';
  Name[WM_RBUTTONDOWN] := 'WM_RBUTTONDOWN';
  Name[WM_RBUTTONUP] := 'WM_RBUTTONUP';
  Name[WM_RBUTTONDBLCLK] := 'WM_RBUTTONDBLCLK';
  Name[WM_MBUTTONDOWN] := 'WM_MBUTTONDOWN';
  Name[WM_MBUTTONUP] := 'WM_MBUTTONUP';
  Name[WM_MBUTTONDBLCLK] := 'WM_MBUTTONDBLCLK';
  Name[WM_MOUSEWHEEL] := 'WM_MOUSEWHEEL';
  Name[WM_XBUTTONDOWN] := 'WM_XBUTTONDOWN';
  Name[WM_XBUTTONUP] := 'WM_XBUTTONUP';
  Name[WM_XBUTTONDBLCLK] := 'WM_XBUTTONDBLCLK';
  Name[WM_MOUSEHWHEEL] := 'WM_MOUSEHWHEEL';
  Name[WM_PARENTNOTIFY] := 'WM_PARENTNOTIFY';
  Name[WM_ENTERMENULOOP] := 'WM_ENTERMENULOOP';
  Name[WM_EXITMENULOOP] := 'WM_EXITMENULOOP';
  Name[WM_NEXTMENU] := 'WM_NEXTMENU';
  Name[WM_SIZING] := 'WM_SIZING';
  Name[WM_CAPTURECHANGED] := 'WM_CAPTURECHANGED';
  Name[WM_MOVING] := 'WM_MOVING';
  Name[WM_POWERBROADCAST] := 'WM_POWERBROADCAST';
  Name[WM_DEVICECHANGE] := 'WM_DEVICECHANGE';
  Name[WM_IME_STARTCOMPOSITION] := 'WM_IME_STARTCOMPOSITION';
  Name[WM_IME_ENDCOMPOSITION] := 'WM_IME_ENDCOMPOSITION';
  Name[WM_IME_COMPOSITION] := 'WM_IME_COMPOSITION';
  Name[WM_IME_SETCONTEXT] := 'WM_IME_SETCONTEXT';
  Name[WM_IME_NOTIFY] := 'WM_IME_NOTIFY';
  Name[WM_IME_CONTROL] := 'WM_IME_CONTROL';
  Name[WM_IME_COMPOSITIONFULL] := 'WM_IME_COMPOSITIONFULL';
  Name[WM_IME_SELECT] := 'WM_IME_SELECT';
  Name[WM_IME_CHAR] := 'WM_IME_CHAR';
  Name[WM_IME_REQUEST] := 'WM_IME_REQUEST';
  Name[WM_IME_KEYDOWN] := 'WM_IME_KEYDOWN';
  Name[WM_IME_KEYUP] := 'WM_IME_KEYUP';
  Name[WM_MDICREATE] := 'WM_MDICREATE';
  Name[WM_MDIDESTROY] := 'WM_MDIDESTROY';
  Name[WM_MDIACTIVATE] := 'WM_MDIACTIVATE';
  Name[WM_MDIRESTORE] := 'WM_MDIRESTORE';
  Name[WM_MDINEXT] := 'WM_MDINEXT';
  Name[WM_MDIMAXIMIZE] := 'WM_MDIMAXIMIZE';
  Name[WM_MDITILE] := 'WM_MDITILE';
  Name[WM_MDICASCADE] := 'WM_MDICASCADE';
  Name[WM_MDIICONARRANGE] := 'WM_MDIICONARRANGE';
  Name[WM_MDIGETACTIVE] := 'WM_MDIGETACTIVE';
  Name[WM_MDISETMENU] := 'WM_MDISETMENU';
  Name[WM_ENTERSIZEMOVE] := 'WM_ENTERSIZEMOVE';
  Name[WM_EXITSIZEMOVE] := 'WM_EXITSIZEMOVE';
  Name[WM_DROPFILES] := 'WM_DROPFILES';
  Name[WM_MDIREFRESHMENU] := 'WM_MDIREFRESHMENU';
  Name[WM_POINTERDEVICECHANGE] := 'WM_POINTERDEVICECHANGE';
  Name[WM_POINTERDEVICEINRANGE] := 'WM_POINTERDEVICEINRANGE';
  Name[WM_POINTERDEVICEOUTOFRANGE] := 'WM_POINTERDEVICEOUTOFRANGE';
  Name[WM_TOUCH] := 'WM_TOUCH';
  Name[WM_NCPOINTERUPDATE] := 'WM_NCPOINTERUPDATE';
  Name[WM_NCPOINTERDOWN] := 'WM_NCPOINTERDOWN';
  Name[WM_NCPOINTERUP] := 'WM_NCPOINTERUP';
  Name[WM_POINTERUPDATE] := 'WM_POINTERUPDATE';
  Name[WM_POINTERDOWN] := 'WM_POINTERDOWN';
  Name[WM_POINTERUP] := 'WM_POINTERUP';
  Name[WM_POINTERENTER] := 'WM_POINTERENTER';
  Name[WM_POINTERLEAVE] := 'WM_POINTERLEAVE';
  Name[WM_POINTERACTIVATE] := 'WM_POINTERACTIVATE';
  Name[WM_POINTERCAPTURECHANGED] := 'WM_POINTERCAPTURECHANGED';
  Name[WM_TOUCHHITTESTING] := 'WM_TOUCHHITTESTING';
  Name[WM_POINTERWHEEL] := 'WM_POINTERWHEEL';
  Name[WM_POINTERHWHEEL] := 'WM_POINTERHWHEEL';
  Name[DM_POINTERHITTEST] := 'DM_POINTERHITTEST';
  Name[WM_POINTERROUTEDTO] := 'WM_POINTERROUTEDTO';
  Name[WM_POINTERROUTEDAWAY] := 'WM_POINTERROUTEDAWAY';
  Name[WM_POINTERROUTEDRELEASED] := 'WM_POINTERROUTEDRELEASED';
  Name[WM_MOUSEHOVER] := 'WM_MOUSEHOVER';
  Name[WM_MOUSELEAVE] := 'WM_MOUSELEAVE';
  Name[WM_NCMOUSEHOVER] := 'WM_NCMOUSEHOVER';
  Name[WM_NCMOUSELEAVE] := 'WM_NCMOUSELEAVE';
  Name[WM_WTSSESSION_CHANGE] := 'WM_WTSSESSION_CHANGE';
  Name[WM_DPICHANGED] := 'WM_DPICHANGED';
  Name[WM_DPICHANGED_BEFOREPARENT] := 'WM_DPICHANGED_BEFOREPARENT';
  Name[WM_DPICHANGED_AFTERPARENT] := 'WM_DPICHANGED_AFTERPARENT';
  Name[WM_GETDPISCALEDSIZE] := 'WM_GETDPISCALEDSIZE';
  Name[WM_CUT] := 'WM_CUT';
  Name[WM_COPY] := 'WM_COPY';
  Name[WM_PASTE] := 'WM_PASTE';
  Name[WM_CLEAR] := 'WM_CLEAR';
  Name[WM_UNDO] := 'WM_UNDO';
  Name[WM_RENDERFORMAT] := 'WM_RENDERFORMAT';
  Name[WM_RENDERALLFORMATS] := 'WM_RENDERALLFORMATS';
  Name[WM_DESTROYCLIPBOARD] := 'WM_DESTROYCLIPBOARD';
  Name[WM_DRAWCLIPBOARD] := 'WM_DRAWCLIPBOARD';
  Name[WM_PAINTCLIPBOARD] := 'WM_PAINTCLIPBOARD';
  Name[WM_VSCROLLCLIPBOARD] := 'WM_VSCROLLCLIPBOARD';
  Name[WM_SIZECLIPBOARD] := 'WM_SIZECLIPBOARD';
  Name[WM_ASKCBFORMATNAME] := 'WM_ASKCBFORMATNAME';
  Name[WM_CHANGECBCHAIN] := 'WM_CHANGECBCHAIN';
  Name[WM_HSCROLLCLIPBOARD] := 'WM_HSCROLLCLIPBOARD';
  Name[WM_QUERYNEWPALETTE] := 'WM_QUERYNEWPALETTE';
  Name[WM_PALETTEISCHANGING] := 'WM_PALETTEISCHANGING';
  Name[WM_PALETTECHANGED] := 'WM_PALETTECHANGED';
  Name[WM_HOTKEY] := 'WM_HOTKEY';
  Name[WM_PRINT] := 'WM_PRINT';
  Name[WM_PRINTCLIENT] := 'WM_PRINTCLIENT';
  Name[WM_APPCOMMAND] := 'WM_APPCOMMAND';
  Name[WM_THEMECHANGED] := 'WM_THEMECHANGED';
  Name[WM_CLIPBOARDUPDATE] := 'WM_CLIPBOARDUPDATE';
  Name[WM_DDE_INITIATE] := 'WM_DDE_INITIATE';
  Name[WM_DDE_TERMINATE] := 'WM_DDE_TERMINATE';
  Name[WM_DDE_UNADVISE] := 'WM_DDE_UNADVISE';
  Name[WM_DWMCOMPOSITIONCHANGED] := 'WM_DWMCOMPOSITIONCHANGED';
  Name[WM_DWMNCRENDERINGCHANGED] := 'WM_DWMNCRENDERINGCHANGED';
  Name[WM_DWMCOLORIZATIONCOLORCHANGED] := 'WM_DWMCOLORIZATIONCOLORCHANGED';
  Name[WM_DWMWINDOWMAXIMIZEDCHANGE] := 'WM_DWMWINDOWMAXIMIZEDCHANGE';
  Name[WM_DWMSENDICONICTHUMBNAIL] := 'WM_DWMSENDICONICTHUMBNAIL';
  Name[WM_DWMSENDICONICLIVEPREVIEWBITMAP] :=
    'WM_DWMSENDICONICLIVEPREVIEWBITMAP';
  Name[WM_GETTITLEBARINFOEX] := 'WM_GETTITLEBARINFOEX';
  Name[WM_TABLET_DEFBASE] := 'WM_TABLET_DEFBASE';
  Name[WM_TABLET_MAXOFFSET] := 'WM_TABLET_MAXOFFSET';
  Name[WM_TABLET_ADDED] := 'WM_TABLET_ADDED';
  Name[WM_TABLET_DELETED] := 'WM_TABLET_DELETED';
  Name[WM_TABLET_FLICK] := 'WM_TABLET_FLICK';
  Name[WM_TABLET_QUERYSYSTEMGESTURESTATUS] :=
    'WM_TABLET_QUERYSYSTEMGESTURESTATUS';
  Name[WM_APP] := 'WM_APP';
  Name[WM_USER] := 'WM_USER';
  Name[BM_GETCHECK] := 'BM_GETCHECK';
  Name[BM_SETCHECK] := 'BM_SETCHECK';
  Name[BM_GETSTATE] := 'BM_GETSTATE';
  Name[BM_SETSTATE] := 'BM_SETSTATE';
  Name[BM_SETSTYLE] := 'BM_SETSTYLE';
  Name[BM_CLICK] := 'BM_CLICK';
  Name[BM_GETIMAGE] := 'BM_GETIMAGE';
  Name[BM_SETIMAGE] := 'BM_SETIMAGE';
  Name[BM_SETDONTCLICK] := 'BM_SETDONTCLICK';
  Name[LB_ADDSTRING] := 'LB_ADDSTRING';
  Name[LB_INSERTSTRING] := 'LB_INSERTSTRING';
  Name[LB_DELETESTRING] := 'LB_DELETESTRING';
  Name[LB_SELITEMRANGEEX] := 'LB_SELITEMRANGEEX';
  Name[LB_RESETCONTENT] := 'LB_RESETCONTENT';
  Name[LB_SETSEL] := 'LB_SETSEL';
  Name[LB_SETCURSEL] := 'LB_SETCURSEL';
  Name[LB_GETSEL] := 'LB_GETSEL';
  Name[LB_GETCURSEL] := 'LB_GETCURSEL';
  Name[LB_GETTEXT] := 'LB_GETTEXT';
  Name[LB_GETTEXTLEN] := 'LB_GETTEXTLEN';
  Name[LB_GETCOUNT] := 'LB_GETCOUNT';
  Name[LB_SELECTSTRING] := 'LB_SELECTSTRING';
  Name[LB_DIR] := 'LB_DIR';
  Name[LB_GETTOPINDEX] := 'LB_GETTOPINDEX';
  Name[LB_FINDSTRING] := 'LB_FINDSTRING';
  Name[LB_GETSELCOUNT] := 'LB_GETSELCOUNT';
  Name[LB_GETSELITEMS] := 'LB_GETSELITEMS';
  Name[LB_SETTABSTOPS] := 'LB_SETTABSTOPS';
  Name[LB_GETHORIZONTALEXTENT] := 'LB_GETHORIZONTALEXTENT';
  Name[LB_SETHORIZONTALEXTENT] := 'LB_SETHORIZONTALEXTENT';
  Name[LB_SETCOLUMNWIDTH] := 'LB_SETCOLUMNWIDTH';
  Name[LB_ADDFILE] := 'LB_ADDFILE';
  Name[LB_SETTOPINDEX] := 'LB_SETTOPINDEX';
  Name[LB_GETITEMRECT] := 'LB_GETITEMRECT';
  Name[LB_GETITEMDATA] := 'LB_GETITEMDATA';
  Name[LB_SETITEMDATA] := 'LB_SETITEMDATA';
  Name[LB_SELITEMRANGE] := 'LB_SELITEMRANGE';
  Name[LB_SETANCHORINDEX] := 'LB_SETANCHORINDEX';
  Name[LB_GETANCHORINDEX] := 'LB_GETANCHORINDEX';
  Name[LB_SETCARETINDEX] := 'LB_SETCARETINDEX';
  Name[LB_GETCARETINDEX] := 'LB_GETCARETINDEX';
  Name[LB_SETITEMHEIGHT] := 'LB_SETITEMHEIGHT';
  Name[LB_GETITEMHEIGHT] := 'LB_GETITEMHEIGHT';
  Name[LB_FINDSTRINGEXACT] := 'LB_FINDSTRINGEXACT';
  Name[LB_SETLOCALE] := 'LB_SETLOCALE';
  Name[LB_GETLOCALE] := 'LB_GETLOCALE';
  Name[LB_SETCOUNT] := 'LB_SETCOUNT';
  Name[LB_INITSTORAGE] := 'LB_INITSTORAGE';
  Name[LB_ITEMFROMPOINT] := 'LB_ITEMFROMPOINT';
  Name[LB_MSGMAX] := 'LB_MSGMAX';
  Name[EM_GETSEL] := 'EM_GETSEL';
  Name[EM_SETSEL] := 'EM_SETSEL';
  Name[EM_GETRECT] := 'EM_GETRECT';
  Name[EM_SETRECT] := 'EM_SETRECT';
  Name[EM_SETRECTNP] := 'EM_SETRECTNP';
  Name[EM_SCROLL] := 'EM_SCROLL';
  Name[EM_LINESCROLL] := 'EM_LINESCROLL';
  Name[EM_SCROLLCARET] := 'EM_SCROLLCARET';
  Name[EM_GETMODIFY] := 'EM_GETMODIFY';
  Name[EM_SETMODIFY] := 'EM_SETMODIFY';
  Name[EM_GETLINECOUNT] := 'EM_GETLINECOUNT';
  Name[EM_LINEINDEX] := 'EM_LINEINDEX';
  Name[EM_SETHANDLE] := 'EM_SETHANDLE';
  Name[EM_GETHANDLE] := 'EM_GETHANDLE';
  Name[EM_GETTHUMB] := 'EM_GETTHUMB';
  Name[EM_LINELENGTH] := 'EM_LINELENGTH';
  Name[EM_REPLACESEL] := 'EM_REPLACESEL';
  Name[EM_GETLINE] := 'EM_GETLINE';
  Name[EM_LIMITTEXT] := 'EM_LIMITTEXT';
  Name[EM_CANUNDO] := 'EM_CANUNDO';
  Name[EM_UNDO] := 'EM_UNDO';
  Name[EM_FMTLINES] := 'EM_FMTLINES';
  Name[EM_LINEFROMCHAR] := 'EM_LINEFROMCHAR';
  Name[EM_SETTABSTOPS] := 'EM_SETTABSTOPS';
  Name[EM_SETPASSWORDCHAR] := 'EM_SETPASSWORDCHAR';
  Name[EM_EMPTYUNDOBUFFER] := 'EM_EMPTYUNDOBUFFER';
  Name[EM_GETFIRSTVISIBLELINE] := 'EM_GETFIRSTVISIBLELINE';
  Name[EM_SETREADONLY] := 'EM_SETREADONLY';
  Name[EM_SETWORDBREAKPROC] := 'EM_SETWORDBREAKPROC';
  Name[EM_GETWORDBREAKPROC] := 'EM_GETWORDBREAKPROC';
  Name[EM_GETPASSWORDCHAR] := 'EM_GETPASSWORDCHAR';
  Name[EM_SETMARGINS] := 'EM_SETMARGINS';
  Name[EM_GETMARGINS] := 'EM_GETMARGINS';
  Name[EM_SETLIMITTEXT] := 'EM_SETLIMITTEXT';
  Name[EM_GETLIMITTEXT] := 'EM_GETLIMITTEXT';
  Name[EM_POSFROMCHAR] := 'EM_POSFROMCHAR';
  Name[EM_CHARFROMPOS] := 'EM_CHARFROMPOS';
  Name[EM_SETIMESTATUS] := 'EM_SETIMESTATUS';
  Name[EM_GETIMESTATUS] := 'EM_GETIMESTATUS';
  Name[EM_ENABLEFEATURE] := 'EM_ENABLEFEATURE';
  Name[SBM_SETPOS] := 'SBM_SETPOS';
  Name[SBM_GETPOS] := 'SBM_GETPOS';
  Name[SBM_SETRANGE] := 'SBM_SETRANGE';
  Name[SBM_SETRANGEREDRAW] := 'SBM_SETRANGEREDRAW';
  Name[SBM_GETRANGE] := 'SBM_GETRANGE';
  Name[SBM_ENABLE_ARROWS] := 'SBM_ENABLE_ARROWS';
  Name[SBM_SETSCROLLINFO] := 'SBM_SETSCROLLINFO';
  Name[SBM_GETSCROLLINFO] := 'SBM_GETSCROLLINFO';
  Name[SBM_GETSCROLLBARINFO] := 'SBM_GETSCROLLBARINFO';
  Name[DM_GETDEFID] := 'DM_GETDEFID';
  Name[DM_SETDEFID] := 'DM_SETDEFID';
  Name[DM_REPOSITION] := 'DM_REPOSITION';
  Name[EM_CANPASTE] := 'EM_CANPASTE';
  Name[EM_EXGETSEL] := 'EM_EXGETSEL';
  Name[EM_EXSETSEL] := 'EM_EXSETSEL';
  Name[EM_FINDTEXT] := 'EM_FINDTEXT';
  Name[EM_GETEVENTMASK] := 'EM_GETEVENTMASK';
  Name[EM_PASTESPECIAL] := 'EM_PASTESPECIAL';
  Name[EM_SETEVENTMASK] := 'EM_SETEVENTMASK';
  Name[EM_STREAMIN] := 'EM_STREAMIN';
  Name[EM_GETTEXTRANGE] := 'EM_GETTEXTRANGE';
  Name[EM_SETUNDOLIMIT] := 'EM_SETUNDOLIMIT';
  Name[EM_REDO] := 'EM_REDO';
  Name[EM_GETAUTOURLDETECT] := 'EM_GETAUTOURLDETECT';
  Name[EM_CONVPOSITION] := 'EM_CONVPOSITION';
  Name[EM_RECONVERSION] := 'EM_RECONVERSION';
  Name[EM_SETTYPOGRAPHYOPTIONS] := 'EM_SETTYPOGRAPHYOPTIONS';
  Name[EM_GETTYPOGRAPHYOPTIONS] := 'EM_GETTYPOGRAPHYOPTIONS';
  Name[EM_SETEDITSTYLE] := 'EM_SETEDITSTYLE';
  Name[EM_GETEDITSTYLE] := 'EM_GETEDITSTYLE';
  Name[EM_GETSCROLLPOS] := 'EM_GETSCROLLPOS';
  Name[EM_SETSCROLLPOS] := 'EM_SETSCROLLPOS';
  Name[EM_GETZOOM] := 'EM_GETZOOM';
  Name[EM_SETZOOM] := 'EM_SETZOOM';
  Name[EM_GETHYPHENATEINFO] := 'EM_GETHYPHENATEINFO';
  Name[EM_SETHYPHENATEINFO] := 'EM_SETHYPHENATEINFO';
  Name[EM_GETCTFOPENSTATUS] := 'EM_GETCTFOPENSTATUS';
  Name[EM_SETCTFOPENSTATUS] := 'EM_SETCTFOPENSTATUS';
  Name[PSM_SETCURSEL] := 'PSM_SETCURSEL';
  Name[PSM_REMOVEPAGE] := 'PSM_REMOVEPAGE';
  Name[PSM_ADDPAGE] := 'PSM_ADDPAGE';
  Name[PSM_CHANGED] := 'PSM_CHANGED';
  Name[PSM_RESTARTWINDOWS] := 'PSM_RESTARTWINDOWS';
  Name[PSM_REBOOTSYSTEM] := 'PSM_REBOOTSYSTEM';
  Name[PSM_CANCELTOCLOSE] := 'PSM_CANCELTOCLOSE';
  Name[PSM_QUERYSIBLINGS] := 'PSM_QUERYSIBLINGS';
  Name[PSM_UNCHANGED] := 'PSM_UNCHANGED';
  Name[PSM_APPLY] := 'PSM_APPLY';
  Name[PSM_SETTITLEA] := 'PSM_SETTITLEA';
  Name[PSM_SETTITLEW] := 'PSM_SETTITLEW';
  Name[PSM_SETTITLE] := 'PSM_SETTITLE';
  Name[PSM_SETWIZBUTTONS] := 'PSM_SETWIZBUTTONS';
  Name[PSM_PRESSBUTTON] := 'PSM_PRESSBUTTON';
  Name[PSM_SETCURSELID] := 'PSM_SETCURSELID';
  Name[PSM_SETFINISHTEXTA] := 'PSM_SETFINISHTEXTA';
  Name[PSM_SETFINISHTEXTW] := 'PSM_SETFINISHTEXTW';
  Name[PSM_SETFINISHTEXT] := 'PSM_SETFINISHTEXT';
  Name[PSM_GETTABCONTROL] := 'PSM_GETTABCONTROL';
  Name[PSM_ISDIALOGMESSAGE] := 'PSM_ISDIALOGMESSAGE';
  Name[PSM_GETCURRENTPAGEHWND] := 'PSM_GETCURRENTPAGEHWND';
  Name[PSM_INSERTPAGE] := 'PSM_INSERTPAGE';
  Name[PSM_SETHEADERTITLEA] := 'PSM_SETHEADERTITLEA';
  Name[PSM_SETHEADERTITLEW] := 'PSM_SETHEADERTITLEW';
  Name[PSM_SETHEADERTITLE] := 'PSM_SETHEADERTITLE';
  Name[PSM_SETHEADERSUBTITLEA] := 'PSM_SETHEADERSUBTITLEA';
  Name[PSM_SETHEADERSUBTITLEW] := 'PSM_SETHEADERSUBTITLEW';
  Name[PSM_SETHEADERSUBTITLE] := 'PSM_SETHEADERSUBTITLE';
  Name[PSM_HWNDTOINDEX] := 'PSM_HWNDTOINDEX';
  Name[PSM_INDEXTOHWND] := 'PSM_INDEXTOHWND';
  Name[PSM_PAGETOINDEX] := 'PSM_PAGETOINDEX';
  Name[PSM_INDEXTOPAGE] := 'PSM_INDEXTOPAGE';
  Name[PSM_IDTOINDEX] := 'PSM_IDTOINDEX';
  Name[PSM_INDEXTOID] := 'PSM_INDEXTOID';
  Name[PSM_GETRESULT] := 'PSM_GETRESULT';
  Name[PSM_RECALCPAGESIZES] := 'PSM_RECALCPAGESIZES';
  Name[PSM_SETNEXTTEXTW] := 'PSM_SETNEXTTEXTW';
  Name[PSM_SETNEXTTEXT] := 'PSM_SETNEXTTEXT';
  Name[PSM_SHOWWIZBUTTONS] := 'PSM_SHOWWIZBUTTONS';
  Name[PSM_ENABLEWIZBUTTONS] := 'PSM_ENABLEWIZBUTTONS';
  Name[PSM_SETBUTTONTEXTW] := 'PSM_SETBUTTONTEXTW';
  Name[PSM_SETBUTTONTEXT] := 'PSM_SETBUTTONTEXT';
  Name[CCM_SETBKCOLOR] := 'CCM_SETBKCOLOR';
  Name[CCM_SETCOLORSCHEME] := 'CCM_SETCOLORSCHEME';
  Name[CCM_GETCOLORSCHEME] := 'CCM_GETCOLORSCHEME';
  Name[CCM_GETDROPTARGET] := 'CCM_GETDROPTARGET';
  Name[CCM_SETUNICODEFORMAT] := 'CCM_SETUNICODEFORMAT';
  Name[CCM_GETUNICODEFORMAT] := 'CCM_GETUNICODEFORMAT';
  Name[CCM_SETVERSION] := 'CCM_SETVERSION';
  Name[CCM_GETVERSION] := 'CCM_GETVERSION';
  Name[CCM_SETNOTIFYWINDOW] := 'CCM_SETNOTIFYWINDOW';
  Name[CCM_SETWINDOWTHEME] := 'CCM_SETWINDOWTHEME';
  Name[CCM_DPISCALE] := 'CCM_DPISCALE';
  Name[MSGF_COMMCTRL_BEGINDRAG] := 'MSGF_COMMCTRL_BEGINDRAG';
  Name[MSGF_COMMCTRL_SIZEHEADER] := 'MSGF_COMMCTRL_SIZEHEADER';
  Name[MSGF_COMMCTRL_DRAGSELECT] := 'MSGF_COMMCTRL_DRAGSELECT';
  Name[MSGF_COMMCTRL_TOOLBARCUST] := 'MSGF_COMMCTRL_TOOLBARCUST';
  Name[HDM_GETITEMCOUNT] := 'HDM_GETITEMCOUNT';
  Name[HDM_INSERTITEMW] := 'HDM_INSERTITEMW';
  Name[HDM_INSERTITEMA] := 'HDM_INSERTITEMA';
  Name[HDM_INSERTITEM] := 'HDM_INSERTITEM';
  Name[HDM_DELETEITEM] := 'HDM_DELETEITEM';
  Name[HDM_GETITEMW] := 'HDM_GETITEMW';
  Name[HDM_GETITEMA] := 'HDM_GETITEMA';
  Name[HDM_GETITEM] := 'HDM_GETITEM';
  Name[HDM_SETITEMA] := 'HDM_SETITEMA';
  Name[HDM_SETITEMW] := 'HDM_SETITEMW';
  Name[HDM_SETITEM] := 'HDM_SETITEM';
  Name[HDM_LAYOUT] := 'HDM_LAYOUT';
  Name[HDM_HITTEST] := 'HDM_HITTEST';
  Name[HDM_GETITEMRECT] := 'HDM_GETITEMRECT';
  Name[HDM_SETIMAGELIST] := 'HDM_SETIMAGELIST';
  Name[HDM_GETIMAGELIST] := 'HDM_GETIMAGELIST';
  Name[HDM_ORDERTOINDEX] := 'HDM_ORDERTOINDEX';
  Name[HDM_CREATEDRAGIMAGE] := 'HDM_CREATEDRAGIMAGE';
  Name[HDM_GETORDERARRAY] := 'HDM_GETORDERARRAY';
  Name[HDM_SETORDERARRAY] := 'HDM_SETORDERARRAY';
  Name[HDM_SETHOTDIVIDER] := 'HDM_SETHOTDIVIDER';
  Name[HDM_SETUNICODEFORMAT] := 'HDM_SETUNICODEFORMAT';
  Name[HDM_GETUNICODEFORMAT] := 'HDM_GETUNICODEFORMAT';
  Name[HDM_SETBITMAPMARGIN] := 'HDM_SETBITMAPMARGIN';
  Name[HDM_GETBITMAPMARGIN] := 'HDM_GETBITMAPMARGIN';
  Name[HDM_SETFILTERCHANGETIMEOUT] := 'HDM_SETFILTERCHANGETIMEOUT';
  Name[HDM_EDITFILTER] := 'HDM_EDITFILTER';
  Name[HDM_CLEARFILTER] := 'HDM_CLEARFILTER';
  Name[HDM_GETITEMDROPDOWNRECT] := 'HDM_GETITEMDROPDOWNRECT';
  Name[HDM_GETOVERFLOWRECT] := 'HDM_GETOVERFLOWRECT';
  Name[HDM_GETFOCUSEDITEM] := 'HDM_GETFOCUSEDITEM';
  Name[HDM_SETFOCUSEDITEM] := 'HDM_SETFOCUSEDITEM';
  Name[RB_INSERTBANDA] := 'RB_INSERTBANDA';
  Name[RB_DELETEBAND] := 'RB_DELETEBAND';
  Name[RB_GETBARINFO] := 'RB_GETBARINFO';
  Name[RB_SETBARINFO] := 'RB_SETBARINFO';
  Name[RB_SETBANDINFOA] := 'RB_SETBANDINFOA';
  Name[RB_SETPARENT] := 'RB_SETPARENT';
  Name[RB_HITTEST] := 'RB_HITTEST';
  Name[RB_GETRECT] := 'RB_GETRECT';
  Name[RB_INSERTBANDW] := 'RB_INSERTBANDW';
  Name[RB_SETBANDINFOW] := 'RB_SETBANDINFOW';
  Name[RB_GETBANDCOUNT] := 'RB_GETBANDCOUNT';
  Name[RB_GETROWCOUNT] := 'RB_GETROWCOUNT';
  Name[RB_GETROWHEIGHT] := 'RB_GETROWHEIGHT';
  Name[RB_IDTOINDEX] := 'RB_IDTOINDEX';
  Name[RB_GETTOOLTIPS] := 'RB_GETTOOLTIPS';
  Name[RB_SETTOOLTIPS] := 'RB_SETTOOLTIPS';
  Name[RB_SETBKCOLOR] := 'RB_SETBKCOLOR';
  Name[RB_GETBKCOLOR] := 'RB_GETBKCOLOR';
  Name[RB_SETTEXTCOLOR] := 'RB_SETTEXTCOLOR';
  Name[RB_GETTEXTCOLOR] := 'RB_GETTEXTCOLOR';
  Name[RB_SIZETORECT] := 'RB_SIZETORECT';
  Name[RB_SETCOLORSCHEME] := 'RB_SETCOLORSCHEME';
  Name[RB_GETCOLORSCHEME] := 'RB_GETCOLORSCHEME';
  Name[RB_BEGINDRAG] := 'RB_BEGINDRAG';
  Name[RB_ENDDRAG] := 'RB_ENDDRAG';
  Name[RB_DRAGMOVE] := 'RB_DRAGMOVE';
  Name[RB_GETBARHEIGHT] := 'RB_GETBARHEIGHT';
  Name[RB_GETBANDINFOW] := 'RB_GETBANDINFOW';
  Name[RB_GETBANDINFOA] := 'RB_GETBANDINFOA';
  Name[RB_MINIMIZEBAND] := 'RB_MINIMIZEBAND';
  Name[RB_MAXIMIZEBAND] := 'RB_MAXIMIZEBAND';
  Name[RB_GETDROPTARGET] := 'RB_GETDROPTARGET';
  Name[RB_GETBANDBORDERS] := 'RB_GETBANDBORDERS';
  Name[RB_SHOWBAND] := 'RB_SHOWBAND';
  Name[RB_SETPALETTE] := 'RB_SETPALETTE';
  Name[RB_GETPALETTE] := 'RB_GETPALETTE';
  Name[RB_MOVEBAND] := 'RB_MOVEBAND';
  Name[RB_SETUNICODEFORMAT] := 'RB_SETUNICODEFORMAT';
  Name[RB_GETUNICODEFORMAT] := 'RB_GETUNICODEFORMAT';
  Name[RB_GETBANDMARGINS] := 'RB_GETBANDMARGINS';
  Name[RB_SETWINDOWTHEME] := 'RB_SETWINDOWTHEME';
  Name[RB_SETEXTENDEDSTYLE] := 'RB_SETEXTENDEDSTYLE';
  Name[RB_GETEXTENDEDSTYLE] := 'RB_GETEXTENDEDSTYLE';
  Name[RB_PUSHCHEVRON] := 'RB_PUSHCHEVRON';
  Name[RB_SETBANDWIDTH] := 'RB_SETBANDWIDTH';
  Name[RB_INSERTBAND] := 'RB_INSERTBAND';
  Name[RB_SETBANDINFO] := 'RB_SETBANDINFO';
  Name[RB_GETBANDINFO] := 'RB_GETBANDINFO';
  Name[TTM_ACTIVATE] := 'TTM_ACTIVATE';
  Name[TTM_SETDELAYTIME] := 'TTM_SETDELAYTIME';
  Name[TTM_ADDTOOLA] := 'TTM_ADDTOOLA';
  Name[TTM_DELTOOLA] := 'TTM_DELTOOLA';
  Name[TTM_NEWTOOLRECTA] := 'TTM_NEWTOOLRECTA';
  Name[TTM_GETTOOLINFOA] := 'TTM_GETTOOLINFOA';
  Name[TTM_SETTOOLINFOA] := 'TTM_SETTOOLINFOA';
  Name[TTM_HITTESTA] := 'TTM_HITTESTA';
  Name[TTM_GETTEXTA] := 'TTM_GETTEXTA';
  Name[TTM_UPDATETIPTEXTA] := 'TTM_UPDATETIPTEXTA';
  Name[TTM_ENUMTOOLSA] := 'TTM_ENUMTOOLSA';
  Name[TTM_GETCURRENTTOOLA] := 'TTM_GETCURRENTTOOLA';
  Name[TTM_ADDTOOLW] := 'TTM_ADDTOOLW';
  Name[TTM_DELTOOLW] := 'TTM_DELTOOLW';
  Name[TTM_NEWTOOLRECTW] := 'TTM_NEWTOOLRECTW';
  Name[TTM_GETTOOLINFOW] := 'TTM_GETTOOLINFOW';
  Name[TTM_SETTOOLINFOW] := 'TTM_SETTOOLINFOW';
  Name[TTM_HITTESTW] := 'TTM_HITTESTW';
  Name[TTM_GETTEXTW] := 'TTM_GETTEXTW';
  Name[TTM_UPDATETIPTEXTW] := 'TTM_UPDATETIPTEXTW';
  Name[TTM_ENUMTOOLSW] := 'TTM_ENUMTOOLSW';
  Name[TTM_GETCURRENTTOOLW] := 'TTM_GETCURRENTTOOLW';
  Name[TTM_WINDOWFROMPOINT] := 'TTM_WINDOWFROMPOINT';
  Name[TTM_TRACKACTIVATE] := 'TTM_TRACKACTIVATE';
  Name[TTM_TRACKPOSITION] := 'TTM_TRACKPOSITION';
  Name[TTM_SETTIPBKCOLOR] := 'TTM_SETTIPBKCOLOR';
  Name[TTM_SETTIPTEXTCOLOR] := 'TTM_SETTIPTEXTCOLOR';
  Name[TTM_GETDELAYTIME] := 'TTM_GETDELAYTIME';
  Name[TTM_GETTIPBKCOLOR] := 'TTM_GETTIPBKCOLOR';
  Name[TTM_GETTIPTEXTCOLOR] := 'TTM_GETTIPTEXTCOLOR';
  Name[TTM_SETMAXTIPWIDTH] := 'TTM_SETMAXTIPWIDTH';
  Name[TTM_GETMAXTIPWIDTH] := 'TTM_GETMAXTIPWIDTH';
  Name[TTM_SETMARGIN] := 'TTM_SETMARGIN';
  Name[TTM_GETMARGIN] := 'TTM_GETMARGIN';
  Name[TTM_POP] := 'TTM_POP';
  Name[TTM_UPDATE] := 'TTM_UPDATE';
  Name[TTM_GETBUBBLESIZE] := 'TTM_GETBUBBLESIZE';
  Name[TTM_ADJUSTRECT] := 'TTM_ADJUSTRECT';
  Name[TTM_SETTITLEA] := 'TTM_SETTITLEA';
  Name[TTM_SETTITLEW] := 'TTM_SETTITLEW';
  Name[TTM_POPUP] := 'TTM_POPUP';
  Name[TTM_GETTITLE] := 'TTM_GETTITLE';
  Name[TTM_ADDTOOL] := 'TTM_ADDTOOL';
  Name[TTM_DELTOOL] := 'TTM_DELTOOL';
  Name[TTM_NEWTOOLRECT] := 'TTM_NEWTOOLRECT';
  Name[TTM_GETTOOLINFO] := 'TTM_GETTOOLINFO';
  Name[TTM_SETTOOLINFO] := 'TTM_SETTOOLINFO';
  Name[TTM_HITTEST] := 'TTM_HITTEST';
  Name[TTM_GETTEXT] := 'TTM_GETTEXT';
  Name[TTM_UPDATETIPTEXT] := 'TTM_UPDATETIPTEXT';
  Name[TTM_ENUMTOOLS] := 'TTM_ENUMTOOLS';
  Name[TTM_GETCURRENTTOOL] := 'TTM_GETCURRENTTOOL';
  Name[TTM_SETTITLE] := 'TTM_SETTITLE';
  Name[TTM_SETWINDOWTHEME] := 'TTM_SETWINDOWTHEME';
  Name[TTM_RELAYEVENT] := 'TTM_RELAYEVENT';
  Name[TTM_GETTOOLCOUNT] := 'TTM_GETTOOLCOUNT';
  Name[SB_SETTEXTA] := 'SB_SETTEXTA';
  Name[SB_GETTEXTA] := 'SB_GETTEXTA';
  Name[SB_GETTEXTLENGTHA] := 'SB_GETTEXTLENGTHA';
  Name[SB_SETTIPTEXTA] := 'SB_SETTIPTEXTA';
  Name[SB_GETTIPTEXTA] := 'SB_GETTIPTEXTA';
  Name[SB_SETTEXTW] := 'SB_SETTEXTW';
  Name[SB_GETTEXTW] := 'SB_GETTEXTW';
  Name[SB_GETTEXTLENGTHW] := 'SB_GETTEXTLENGTHW';
  Name[SB_SETTIPTEXTW] := 'SB_SETTIPTEXTW';
  Name[SB_GETTIPTEXTW] := 'SB_GETTIPTEXTW';
  Name[SB_SETTEXT] := 'SB_SETTEXT';
  Name[SB_GETTEXT] := 'SB_GETTEXT';
  Name[SB_GETTEXTLENGTH] := 'SB_GETTEXTLENGTH';
  Name[SB_SETTIPTEXT] := 'SB_SETTIPTEXT';
  Name[SB_GETTIPTEXT] := 'SB_GETTIPTEXT';
  Name[SB_SETPARTS] := 'SB_SETPARTS';
  Name[SB_GETPARTS] := 'SB_GETPARTS';
  Name[SB_GETBORDERS] := 'SB_GETBORDERS';
  Name[SB_SETMINHEIGHT] := 'SB_SETMINHEIGHT';
  Name[SB_SIMPLE] := 'SB_SIMPLE';
  Name[SB_GETRECT] := 'SB_GETRECT';
  Name[SB_ISSIMPLE] := 'SB_ISSIMPLE';
  Name[SB_SETICON] := 'SB_SETICON';
  Name[SB_GETICON] := 'SB_GETICON';
  Name[SB_SETUNICODEFORMAT] := 'SB_SETUNICODEFORMAT';
  Name[SB_GETUNICODEFORMAT] := 'SB_GETUNICODEFORMAT';
  Name[SB_SETBKCOLOR] := 'SB_SETBKCOLOR';
  Name[TBM_GETPOS] := 'TBM_GETPOS';
  Name[TBM_GETRANGEMIN] := 'TBM_GETRANGEMIN';
  Name[TBM_GETRANGEMAX] := 'TBM_GETRANGEMAX';
  Name[TBM_GETTIC] := 'TBM_GETTIC';
  Name[TBM_SETTIC] := 'TBM_SETTIC';
  Name[TBM_SETPOS] := 'TBM_SETPOS';
  Name[TBM_SETRANGE] := 'TBM_SETRANGE';
  Name[TBM_SETRANGEMIN] := 'TBM_SETRANGEMIN';
  Name[TBM_SETRANGEMAX] := 'TBM_SETRANGEMAX';
  Name[TBM_CLEARTICS] := 'TBM_CLEARTICS';
  Name[TBM_SETSEL] := 'TBM_SETSEL';
  Name[TBM_SETSELSTART] := 'TBM_SETSELSTART';
  Name[TBM_SETSELEND] := 'TBM_SETSELEND';
  Name[TBM_GETPTICS] := 'TBM_GETPTICS';
  Name[TBM_GETTICPOS] := 'TBM_GETTICPOS';
  Name[TBM_GETNUMTICS] := 'TBM_GETNUMTICS';
  Name[TBM_GETSELSTART] := 'TBM_GETSELSTART';
  Name[TBM_GETSELEND] := 'TBM_GETSELEND';
  Name[TBM_CLEARSEL] := 'TBM_CLEARSEL';
  Name[TBM_SETTICFREQ] := 'TBM_SETTICFREQ';
  Name[TBM_SETPAGESIZE] := 'TBM_SETPAGESIZE';
  Name[TBM_GETPAGESIZE] := 'TBM_GETPAGESIZE';
  Name[TBM_SETLINESIZE] := 'TBM_SETLINESIZE';
  Name[TBM_GETLINESIZE] := 'TBM_GETLINESIZE';
  Name[TBM_GETTHUMBRECT] := 'TBM_GETTHUMBRECT';
  Name[TBM_GETCHANNELRECT] := 'TBM_GETCHANNELRECT';
  Name[TBM_SETTHUMBLENGTH] := 'TBM_SETTHUMBLENGTH';
  Name[TBM_GETTHUMBLENGTH] := 'TBM_GETTHUMBLENGTH';
  Name[TBM_SETTOOLTIPS] := 'TBM_SETTOOLTIPS';
  Name[TBM_GETTOOLTIPS] := 'TBM_GETTOOLTIPS';
  Name[TBM_SETTIPSIDE] := 'TBM_SETTIPSIDE';
  Name[TBM_SETBUDDY] := 'TBM_SETBUDDY';
  Name[TBM_GETBUDDY] := 'TBM_GETBUDDY';
  Name[TBM_SETPOSNOTIFY] := 'TBM_SETPOSNOTIFY';
  Name[TBM_SETUNICODEFORMAT] := 'TBM_SETUNICODEFORMAT';
  Name[TBM_GETUNICODEFORMAT] := 'TBM_GETUNICODEFORMAT';
  Name[UDM_SETRANGE] := 'UDM_SETRANGE';
  Name[UDM_GETRANGE] := 'UDM_GETRANGE';
  Name[UDM_SETPOS] := 'UDM_SETPOS';
  Name[UDM_GETPOS] := 'UDM_GETPOS';
  Name[UDM_SETBUDDY] := 'UDM_SETBUDDY';
  Name[UDM_GETBUDDY] := 'UDM_GETBUDDY';
  Name[UDM_SETACCEL] := 'UDM_SETACCEL';
  Name[UDM_GETACCEL] := 'UDM_GETACCEL';
  Name[UDM_SETBASE] := 'UDM_SETBASE';
  Name[UDM_GETBASE] := 'UDM_GETBASE';
  Name[UDM_SETRANGE32] := 'UDM_SETRANGE32';
  Name[UDM_GETRANGE32] := 'UDM_GETRANGE32';
  Name[UDM_SETUNICODEFORMAT] := 'UDM_SETUNICODEFORMAT';
  Name[UDM_GETUNICODEFORMAT] := 'UDM_GETUNICODEFORMAT';
  Name[UDM_SETPOS32] := 'UDM_SETPOS32';
  Name[UDM_GETPOS32] := 'UDM_GETPOS32';
  Name[PBM_SETRANGE] := 'PBM_SETRANGE';
  Name[PBM_SETPOS] := 'PBM_SETPOS';
  Name[PBM_DELTAPOS] := 'PBM_DELTAPOS';
  Name[PBM_SETSTEP] := 'PBM_SETSTEP';
  Name[PBM_STEPIT] := 'PBM_STEPIT';
  Name[PBM_SETRANGE32] := 'PBM_SETRANGE32';
  Name[PBM_GETRANGE] := 'PBM_GETRANGE';
  Name[PBM_GETPOS] := 'PBM_GETPOS';
  Name[PBM_SETBARCOLOR] := 'PBM_SETBARCOLOR';
  Name[PBM_SETBKCOLOR] := 'PBM_SETBKCOLOR';
  Name[PBM_SETMARQUEE] := 'PBM_SETMARQUEE';
  Name[PBM_GETSTEP] := 'PBM_GETSTEP';
  Name[PBM_GETBKCOLOR] := 'PBM_GETBKCOLOR';
  Name[PBM_GETBARCOLOR] := 'PBM_GETBARCOLOR';
  Name[PBM_SETSTATE] := 'PBM_SETSTATE';
  Name[PBM_GETSTATE] := 'PBM_GETSTATE';
  Name[HKM_SETHOTKEY] := 'HKM_SETHOTKEY';
  Name[HKM_GETHOTKEY] := 'HKM_GETHOTKEY';
  Name[HKM_SETRULES] := 'HKM_SETRULES';
  Name[LM_HITTEST] := 'LM_HITTEST';
  Name[LM_GETIDEALHEIGHT] := 'LM_GETIDEALHEIGHT';
  Name[LM_SETITEM] := 'LM_SETITEM';
  Name[LM_GETITEM] := 'LM_GETITEM';
  Name[LM_GETIDEALSIZE] := 'LM_GETIDEALSIZE';
  Name[LVM_SETUNICODEFORMAT] := 'LVM_SETUNICODEFORMAT';
  Name[LVM_GETUNICODEFORMAT] := 'LVM_GETUNICODEFORMAT';
  Name[LVM_GETBKCOLOR] := 'LVM_GETBKCOLOR';
  Name[LVM_SETBKCOLOR] := 'LVM_SETBKCOLOR';
  Name[LVM_GETIMAGELIST] := 'LVM_GETIMAGELIST';
  Name[LVM_SETIMAGELIST] := 'LVM_SETIMAGELIST';
  Name[LVM_GETITEMCOUNT] := 'LVM_GETITEMCOUNT';
  Name[LVM_GETITEMA] := 'LVM_GETITEMA';
  Name[LVM_SETITEMA] := 'LVM_SETITEMA';
  Name[LVM_INSERTITEMA] := 'LVM_INSERTITEMA';
  Name[LVM_GETITEMW] := 'LVM_GETITEMW';
  Name[LVM_SETITEMW] := 'LVM_SETITEMW';
  Name[LVM_INSERTITEMW] := 'LVM_INSERTITEMW';
  Name[LVM_GETITEM] := 'LVM_GETITEM';
  Name[LVM_SETITEM] := 'LVM_SETITEM';
  Name[LVM_INSERTITEM] := 'LVM_INSERTITEM';
  Name[LVM_DELETEITEM] := 'LVM_DELETEITEM';
  Name[LVM_DELETEALLITEMS] := 'LVM_DELETEALLITEMS';
  Name[LVM_GETCALLBACKMASK] := 'LVM_GETCALLBACKMASK';
  Name[LVM_SETCALLBACKMASK] := 'LVM_SETCALLBACKMASK';
  Name[LVM_GETNEXTITEM] := 'LVM_GETNEXTITEM';
  Name[LVM_FINDITEMA] := 'LVM_FINDITEMA';
  Name[LVM_FINDITEMW] := 'LVM_FINDITEMW';
  Name[LVM_FINDITEM] := 'LVM_FINDITEM';
  Name[LVM_GETITEMRECT] := 'LVM_GETITEMRECT';
  Name[LVM_SETITEMPOSITION] := 'LVM_SETITEMPOSITION';
  Name[LVM_GETITEMPOSITION] := 'LVM_GETITEMPOSITION';
  Name[LVM_GETSTRINGWIDTHA] := 'LVM_GETSTRINGWIDTHA';
  Name[LVM_GETSTRINGWIDTHW] := 'LVM_GETSTRINGWIDTHW';
  Name[LVM_GETSTRINGWIDTH] := 'LVM_GETSTRINGWIDTH';
  Name[LVM_HITTEST] := 'LVM_HITTEST';
  Name[LVM_ENSUREVISIBLE] := 'LVM_ENSUREVISIBLE';
  Name[LVM_SCROLL] := 'LVM_SCROLL';
  Name[LVM_REDRAWITEMS] := 'LVM_REDRAWITEMS';
  Name[LVM_ARRANGE] := 'LVM_ARRANGE';
  Name[LVM_EDITLABELA] := 'LVM_EDITLABELA';
  Name[LVM_EDITLABELW] := 'LVM_EDITLABELW';
  Name[LVM_EDITLABEL] := 'LVM_EDITLABEL';
  Name[LVM_GETEDITCONTROL] := 'LVM_GETEDITCONTROL';
  Name[LVM_GETCOLUMNA] := 'LVM_GETCOLUMNA';
  Name[LVM_GETCOLUMNW] := 'LVM_GETCOLUMNW';
  Name[LVM_GETCOLUMN] := 'LVM_GETCOLUMN';
  Name[LVM_SETCOLUMNA] := 'LVM_SETCOLUMNA';
  Name[LVM_SETCOLUMNW] := 'LVM_SETCOLUMNW';
  Name[LVM_SETCOLUMN] := 'LVM_SETCOLUMN';
  Name[LVM_INSERTCOLUMNA] := 'LVM_INSERTCOLUMNA';
  Name[LVM_INSERTCOLUMNW] := 'LVM_INSERTCOLUMNW';
  Name[LVM_DELETECOLUMN] := 'LVM_DELETECOLUMN';
  Name[LVM_GETCOLUMNWIDTH] := 'LVM_GETCOLUMNWIDTH';
  Name[LVM_SETCOLUMNWIDTH] := 'LVM_SETCOLUMNWIDTH';
  Name[LVM_GETHEADER] := 'LVM_GETHEADER';
  Name[LVM_CREATEDRAGIMAGE] := 'LVM_CREATEDRAGIMAGE';
  Name[LVM_GETVIEWRECT] := 'LVM_GETVIEWRECT';
  Name[LVM_GETTEXTCOLOR] := 'LVM_GETTEXTCOLOR';
  Name[LVM_SETTEXTCOLOR] := 'LVM_SETTEXTCOLOR';
  Name[LVM_GETTEXTBKCOLOR] := 'LVM_GETTEXTBKCOLOR';
  Name[LVM_SETTEXTBKCOLOR] := 'LVM_SETTEXTBKCOLOR';
  Name[LVM_GETTOPINDEX] := 'LVM_GETTOPINDEX';
  Name[LVM_GETCOUNTPERPAGE] := 'LVM_GETCOUNTPERPAGE';
  Name[LVM_GETORIGIN] := 'LVM_GETORIGIN';
  Name[LVM_UPDATE] := 'LVM_UPDATE';
  Name[LVM_SETITEMSTATE] := 'LVM_SETITEMSTATE';
  Name[LVM_GETITEMSTATE] := 'LVM_GETITEMSTATE';
  Name[LVM_GETITEMTEXTA] := 'LVM_GETITEMTEXTA';
  Name[LVM_GETITEMTEXTW] := 'LVM_GETITEMTEXTW';
  Name[LVM_GETITEMTEXT] := 'LVM_GETITEMTEXT';
  Name[LVM_SETITEMTEXTA] := 'LVM_SETITEMTEXTA';
  Name[LVM_SETITEMTEXTW] := 'LVM_SETITEMTEXTW';
  Name[LVM_SETITEMTEXT] := 'LVM_SETITEMTEXT';
  Name[LVM_SETITEMCOUNT] := 'LVM_SETITEMCOUNT';
  Name[LVM_SORTITEMS] := 'LVM_SORTITEMS';
  Name[LVM_SETITEMPOSITION32] := 'LVM_SETITEMPOSITION32';
  Name[LVM_GETSELECTEDCOUNT] := 'LVM_GETSELECTEDCOUNT';
  Name[LVM_GETITEMSPACING] := 'LVM_GETITEMSPACING';
  Name[LVM_GETISEARCHSTRINGA] := 'LVM_GETISEARCHSTRINGA';
  Name[LVM_GETISEARCHSTRINGW] := 'LVM_GETISEARCHSTRINGW';
  Name[LVM_GETISEARCHSTRING] := 'LVM_GETISEARCHSTRING';
  Name[LVM_SETICONSPACING] := 'LVM_SETICONSPACING';
  Name[LVM_SETEXTENDEDLISTVIEWSTYLE] := 'LVM_SETEXTENDEDLISTVIEWSTYLE';
  Name[LVM_GETEXTENDEDLISTVIEWSTYLE] := 'LVM_GETEXTENDEDLISTVIEWSTYLE';
  Name[LVM_GETSUBITEMRECT] := 'LVM_GETSUBITEMRECT';
  Name[LVM_SUBITEMHITTEST] := 'LVM_SUBITEMHITTEST';
  Name[LVM_SETCOLUMNORDERARRAY] := 'LVM_SETCOLUMNORDERARRAY';
  Name[LVM_GETCOLUMNORDERARRAY] := 'LVM_GETCOLUMNORDERARRAY';
  Name[LVM_SETHOTITEM] := 'LVM_SETHOTITEM';
  Name[LVM_GETHOTITEM] := 'LVM_GETHOTITEM';
  Name[LVM_SETHOTCURSOR] := 'LVM_SETHOTCURSOR';
  Name[LVM_GETHOTCURSOR] := 'LVM_GETHOTCURSOR';
  Name[LVM_APPROXIMATEVIEWRECT] := 'LVM_APPROXIMATEVIEWRECT';
  Name[LVM_GETSELECTIONMARK] := 'LVM_GETSELECTIONMARK';
  Name[LVM_SETSELECTIONMARK] := 'LVM_SETSELECTIONMARK';
  Name[LVM_GETWORKAREAS] := 'LVM_GETWORKAREAS';
  Name[LVM_SETHOVERTIME] := 'LVM_SETHOVERTIME';
  Name[LVM_GETHOVERTIME] := 'LVM_GETHOVERTIME';
  Name[LVM_GETNUMBEROFWORKAREAS] := 'LVM_GETNUMBEROFWORKAREAS';
  Name[LVM_SETTOOLTIPS] := 'LVM_SETTOOLTIPS';
  Name[LVM_GETTOOLTIPS] := 'LVM_GETTOOLTIPS';
  Name[LVM_SORTITEMSEX] := 'LVM_SORTITEMSEX';
  Name[LVM_SETBKIMAGEA] := 'LVM_SETBKIMAGEA';
  Name[LVM_SETBKIMAGEW] := 'LVM_SETBKIMAGEW';
  Name[LVM_GETBKIMAGEA] := 'LVM_GETBKIMAGEA';
  Name[LVM_GETBKIMAGEW] := 'LVM_GETBKIMAGEW';
  Name[LVM_SETSELECTEDCOLUMN] := 'LVM_SETSELECTEDCOLUMN';
  Name[LVM_SETVIEW] := 'LVM_SETVIEW';
  Name[LVM_GETVIEW] := 'LVM_GETVIEW';
  Name[LVM_INSERTGROUP] := 'LVM_INSERTGROUP';
  Name[LVM_SETGROUPINFO] := 'LVM_SETGROUPINFO';
  Name[LVM_GETGROUPINFO] := 'LVM_GETGROUPINFO';
  Name[LVM_REMOVEGROUP] := 'LVM_REMOVEGROUP';
  Name[LVM_MOVEGROUP] := 'LVM_MOVEGROUP';
  Name[LVM_GETGROUPCOUNT] := 'LVM_GETGROUPCOUNT';
  Name[LVM_GETGROUPINFOBYINDEX] := 'LVM_GETGROUPINFOBYINDEX';
  Name[LVM_MOVEITEMTOGROUP] := 'LVM_MOVEITEMTOGROUP';
  Name[LVM_GETGROUPRECT] := 'LVM_GETGROUPRECT';
  Name[LVM_SETGROUPMETRICS] := 'LVM_SETGROUPMETRICS';
  Name[LVM_GETGROUPMETRICS] := 'LVM_GETGROUPMETRICS';
  Name[LVM_ENABLEGROUPVIEW] := 'LVM_ENABLEGROUPVIEW';
  Name[LVM_SORTGROUPS] := 'LVM_SORTGROUPS';
  Name[LVM_INSERTGROUPSORTED] := 'LVM_INSERTGROUPSORTED';
  Name[LVM_REMOVEALLGROUPS] := 'LVM_REMOVEALLGROUPS';
  Name[LVM_HASGROUP] := 'LVM_HASGROUP';
  Name[LVM_GETGROUPSTATE] := 'LVM_GETGROUPSTATE';
  Name[LVM_GETFOCUSEDGROUP] := 'LVM_GETFOCUSEDGROUP';
  Name[LVM_SETTILEVIEWINFO] := 'LVM_SETTILEVIEWINFO';
  Name[LVM_GETTILEVIEWINFO] := 'LVM_GETTILEVIEWINFO';
  Name[LVM_SETTILEINFO] := 'LVM_SETTILEINFO';
  Name[LVM_GETTILEINFO] := 'LVM_GETTILEINFO';
  Name[LVM_SETINSERTMARK] := 'LVM_SETINSERTMARK';
  Name[LVM_GETINSERTMARK] := 'LVM_GETINSERTMARK';
  Name[LVM_INSERTMARKHITTEST] := 'LVM_INSERTMARKHITTEST';
  Name[LVM_GETINSERTMARKRECT] := 'LVM_GETINSERTMARKRECT';
  Name[LVM_SETINSERTMARKCOLOR] := 'LVM_SETINSERTMARKCOLOR';
  Name[LVM_GETINSERTMARKCOLOR] := 'LVM_GETINSERTMARKCOLOR';
  Name[LVM_SETINFOTIP] := 'LVM_SETINFOTIP';
  Name[LVM_GETSELECTEDCOLUMN] := 'LVM_GETSELECTEDCOLUMN';
  Name[LVM_ISGROUPVIEWENABLED] := 'LVM_ISGROUPVIEWENABLED';
  Name[LVM_GETOUTLINECOLOR] := 'LVM_GETOUTLINECOLOR';
  Name[LVM_SETOUTLINECOLOR] := 'LVM_SETOUTLINECOLOR';
  Name[LVM_CANCELEDITLABEL] := 'LVM_CANCELEDITLABEL';
  Name[LVM_MAPINDEXTOID] := 'LVM_MAPINDEXTOID';
  Name[LVM_MAPIDTOINDEX] := 'LVM_MAPIDTOINDEX';
  Name[LVM_ISITEMVISIBLE] := 'LVM_ISITEMVISIBLE';
  Name[LVM_GETEMPTYTEXT] := 'LVM_GETEMPTYTEXT';
  Name[LVM_GETFOOTERRECT] := 'LVM_GETFOOTERRECT';
  Name[LVM_GETFOOTERINFO] := 'LVM_GETFOOTERINFO';
  Name[LVM_GETFOOTERITEMRECT] := 'LVM_GETFOOTERITEMRECT';
  Name[LVM_GETFOOTERITEM] := 'LVM_GETFOOTERITEM';
  Name[LVM_GETITEMINDEXRECT] := 'LVM_GETITEMINDEXRECT';
  Name[LVM_SETITEMINDEXSTATE] := 'LVM_SETITEMINDEXSTATE';
  Name[LVM_GETNEXTITEMINDEX] := 'LVM_GETNEXTITEMINDEX';
  Name[LVM_SETBKIMAGE] := 'LVM_SETBKIMAGE';
  Name[LVM_GETBKIMAGE] := 'LVM_GETBKIMAGE';
  Name[CBEM_INSERTITEMA] := 'CBEM_INSERTITEMA';
  Name[CBEM_SETIMAGELIST] := 'CBEM_SETIMAGELIST';
  Name[CBEM_GETIMAGELIST] := 'CBEM_GETIMAGELIST';
  Name[CBEM_GETITEMA] := 'CBEM_GETITEMA';
  Name[CBEM_SETITEMA] := 'CBEM_SETITEMA';
  Name[CBEM_DELETEITEM] := 'CBEM_DELETEITEM';
  Name[CBEM_GETCOMBOCONTROL] := 'CBEM_GETCOMBOCONTROL';
  Name[CBEM_GETEDITCONTROL] := 'CBEM_GETEDITCONTROL';
  Name[CBEM_SETEXSTYLE] := 'CBEM_SETEXSTYLE';
  Name[CBEM_GETEXSTYLE] := 'CBEM_GETEXSTYLE';
  Name[CBEM_GETEXTENDEDSTYLE] := 'CBEM_GETEXTENDEDSTYLE';
  Name[CBEM_HASEDITCHANGED] := 'CBEM_HASEDITCHANGED';
  Name[CBEM_INSERTITEMW] := 'CBEM_INSERTITEMW';
  Name[CBEM_SETITEMW] := 'CBEM_SETITEMW';
  Name[CBEM_GETITEMW] := 'CBEM_GETITEMW';
  Name[CBEM_SETEXTENDEDSTYLE] := 'CBEM_SETEXTENDEDSTYLE';
  Name[CBEM_SETUNICODEFORMAT] := 'CBEM_SETUNICODEFORMAT';
  Name[CBEM_GETUNICODEFORMAT] := 'CBEM_GETUNICODEFORMAT';
  Name[CBEM_SETWINDOWTHEME] := 'CBEM_SETWINDOWTHEME';
  Name[CBEM_INSERTITEM] := 'CBEM_INSERTITEM';
  Name[CBEM_SETITEM] := 'CBEM_SETITEM';
  Name[CBEM_GETITEM] := 'CBEM_GETITEM';
  Name[TCM_GETIMAGELIST] := 'TCM_GETIMAGELIST';
  Name[TCM_SETIMAGELIST] := 'TCM_SETIMAGELIST';
  Name[TCM_GETITEMCOUNT] := 'TCM_GETITEMCOUNT';
  Name[TCM_DELETEITEM] := 'TCM_DELETEITEM';
  Name[TCM_DELETEALLITEMS] := 'TCM_DELETEALLITEMS';
  Name[TCM_GETITEMRECT] := 'TCM_GETITEMRECT';
  Name[TCM_GETCURSEL] := 'TCM_GETCURSEL';
  Name[TCM_SETCURSEL] := 'TCM_SETCURSEL';
  Name[TCM_HITTEST] := 'TCM_HITTEST';
  Name[TCM_SETITEMEXTRA] := 'TCM_SETITEMEXTRA';
  Name[TCM_ADJUSTRECT] := 'TCM_ADJUSTRECT';
  Name[TCM_SETITEMSIZE] := 'TCM_SETITEMSIZE';
  Name[TCM_REMOVEIMAGE] := 'TCM_REMOVEIMAGE';
  Name[TCM_SETPADDING] := 'TCM_SETPADDING';
  Name[TCM_GETROWCOUNT] := 'TCM_GETROWCOUNT';
  Name[TCM_GETTOOLTIPS] := 'TCM_GETTOOLTIPS';
  Name[TCM_SETTOOLTIPS] := 'TCM_SETTOOLTIPS';
  Name[TCM_GETCURFOCUS] := 'TCM_GETCURFOCUS';
  Name[TCM_SETCURFOCUS] := 'TCM_SETCURFOCUS';
  Name[TCM_SETMINTABWIDTH] := 'TCM_SETMINTABWIDTH';
  Name[TCM_DESELECTALL] := 'TCM_DESELECTALL';
  Name[TCM_HIGHLIGHTITEM] := 'TCM_HIGHLIGHTITEM';
  Name[TCM_SETEXTENDEDSTYLE] := 'TCM_SETEXTENDEDSTYLE';
  Name[TCM_GETEXTENDEDSTYLE] := 'TCM_GETEXTENDEDSTYLE';
  Name[TCM_SETUNICODEFORMAT] := 'TCM_SETUNICODEFORMAT';
  Name[TCM_GETUNICODEFORMAT] := 'TCM_GETUNICODEFORMAT';
  Name[TCM_GETITEMA] := 'TCM_GETITEMA';
  Name[TCM_SETITEMA] := 'TCM_SETITEMA';
  Name[TCM_INSERTITEMA] := 'TCM_INSERTITEMA';
  Name[TCM_GETITEMW] := 'TCM_GETITEMW';
  Name[TCM_SETITEMW] := 'TCM_SETITEMW';
  Name[TCM_INSERTITEMW] := 'TCM_INSERTITEMW';
  Name[TCM_GETITEM] := 'TCM_GETITEM';
  Name[TCM_SETITEM] := 'TCM_SETITEM';
  Name[TCM_INSERTITEM] := 'TCM_INSERTITEM';
  Name[ACM_OPENA] := 'ACM_OPENA';
  Name[ACM_OPENW] := 'ACM_OPENW';
  Name[ACM_OPEN] := 'ACM_OPEN';
  Name[ACM_PLAY] := 'ACM_PLAY';
  Name[ACM_STOP] := 'ACM_STOP';
  Name[MCM_GETCURSEL] := 'MCM_GETCURSEL';
  Name[MCM_SETCURSEL] := 'MCM_SETCURSEL';
  Name[MCM_GETMAXSELCOUNT] := 'MCM_GETMAXSELCOUNT';
  Name[MCM_SETMAXSELCOUNT] := 'MCM_SETMAXSELCOUNT';
  Name[MCM_GETSELRANGE] := 'MCM_GETSELRANGE';
  Name[MCM_SETSELRANGE] := 'MCM_SETSELRANGE';
  Name[MCM_GETMONTHRANGE] := 'MCM_GETMONTHRANGE';
  Name[MCM_SETDAYSTATE] := 'MCM_SETDAYSTATE';
  Name[MCM_GETMINREQRECT] := 'MCM_GETMINREQRECT';
  Name[MCM_SETCOLOR] := 'MCM_SETCOLOR';
  Name[MCM_GETCOLOR] := 'MCM_GETCOLOR';
  Name[MCM_SETTODAY] := 'MCM_SETTODAY';
  Name[MCM_GETTODAY] := 'MCM_GETTODAY';
  Name[MCM_HITTEST] := 'MCM_HITTEST';
  Name[MCM_SETFIRSTDAYOFWEEK] := 'MCM_SETFIRSTDAYOFWEEK';
  Name[MCM_GETFIRSTDAYOFWEEK] := 'MCM_GETFIRSTDAYOFWEEK';
  Name[MCM_GETRANGE] := 'MCM_GETRANGE';
  Name[MCM_SETRANGE] := 'MCM_SETRANGE';
  Name[MCM_GETMONTHDELTA] := 'MCM_GETMONTHDELTA';
  Name[MCM_SETMONTHDELTA] := 'MCM_SETMONTHDELTA';
  Name[MCM_GETMAXTODAYWIDTH] := 'MCM_GETMAXTODAYWIDTH';
  Name[MCM_SETUNICODEFORMAT] := 'MCM_SETUNICODEFORMAT';
  Name[MCM_GETUNICODEFORMAT] := 'MCM_GETUNICODEFORMAT';
  Name[MCM_GETCURRENTVIEW] := 'MCM_GETCURRENTVIEW';
  Name[MCM_GETCALENDARCOUNT] := 'MCM_GETCALENDARCOUNT';
  Name[MCM_GETCALENDARGRIDINFO] := 'MCM_GETCALENDARGRIDINFO';
  Name[MCM_GETCALID] := 'MCM_GETCALID';
  Name[MCM_SETCALID] := 'MCM_SETCALID';
  Name[MCM_SIZERECTTOMIN] := 'MCM_SIZERECTTOMIN';
  Name[MCM_SETCALENDARBORDER] := 'MCM_SETCALENDARBORDER';
  Name[MCM_GETCALENDARBORDER] := 'MCM_GETCALENDARBORDER';
  Name[MCM_SETCURRENTVIEW] := 'MCM_SETCURRENTVIEW';
  Name[DTM_GETSYSTEMTIME] := 'DTM_GETSYSTEMTIME';
  Name[DTM_SETSYSTEMTIME] := 'DTM_SETSYSTEMTIME';
  Name[DTM_GETRANGE] := 'DTM_GETRANGE';
  Name[DTM_SETRANGE] := 'DTM_SETRANGE';
  Name[DTM_SETFORMATA] := 'DTM_SETFORMATA';
  Name[DTM_SETMCCOLOR] := 'DTM_SETMCCOLOR';
  Name[DTM_GETMCCOLOR] := 'DTM_GETMCCOLOR';
  Name[DTM_GETMONTHCAL] := 'DTM_GETMONTHCAL';
  Name[DTM_SETMCFONT] := 'DTM_SETMCFONT';
  Name[DTM_GETMCFONT] := 'DTM_GETMCFONT';
  Name[DTM_SETFORMATW] := 'DTM_SETFORMATW';
  Name[DTM_SETFORMAT] := 'DTM_SETFORMAT';
  Name[DTM_SETMCSTYLE] := 'DTM_SETMCSTYLE';
  Name[DTM_GETMCSTYLE] := 'DTM_GETMCSTYLE';
  Name[DTM_CLOSEMONTHCAL] := 'DTM_CLOSEMONTHCAL';
  Name[DTM_GETDATETIMEPICKERINFO] := 'DTM_GETDATETIMEPICKERINFO';
  Name[DTM_GETIDEALSIZE] := 'DTM_GETIDEALSIZE';
  Name[IPM_CLEARADDRESS] := 'IPM_CLEARADDRESS';
  Name[IPM_SETADDRESS] := 'IPM_SETADDRESS';
  Name[IPM_GETADDRESS] := 'IPM_GETADDRESS';
  Name[IPM_SETRANGE] := 'IPM_SETRANGE';
  Name[IPM_SETFOCUS] := 'IPM_SETFOCUS';
  Name[IPM_ISBLANK] := 'IPM_ISBLANK';
  Name[PGM_SETCHILD] := 'PGM_SETCHILD';
  Name[PGM_RECALCSIZE] := 'PGM_RECALCSIZE';
  Name[PGM_FORWARDMOUSE] := 'PGM_FORWARDMOUSE';
  Name[PGM_SETBKCOLOR] := 'PGM_SETBKCOLOR';
  Name[PGM_GETBKCOLOR] := 'PGM_GETBKCOLOR';
  Name[PGM_SETBORDER] := 'PGM_SETBORDER';
  Name[PGM_GETBORDER] := 'PGM_GETBORDER';
  Name[PGM_SETPOS] := 'PGM_SETPOS';
  Name[PGM_GETPOS] := 'PGM_GETPOS';
  Name[PGM_SETBUTTONSIZE] := 'PGM_SETBUTTONSIZE';
  Name[PGM_GETBUTTONSIZE] := 'PGM_GETBUTTONSIZE';
  Name[PGM_GETBUTTONSTATE] := 'PGM_GETBUTTONSTATE';
  Name[PGM_GETDROPTARGET] := 'PGM_GETDROPTARGET';
  Name[PGM_SETSCROLLINFO] := 'PGM_SETSCROLLINFO';
  Name[BCM_GETIDEALSIZE] := 'BCM_GETIDEALSIZE';
  Name[BCM_SETIMAGELIST] := 'BCM_SETIMAGELIST';
  Name[BCM_GETIMAGELIST] := 'BCM_GETIMAGELIST';
  Name[BCM_SETTEXTMARGIN] := 'BCM_SETTEXTMARGIN';
  Name[BCM_GETTEXTMARGIN] := 'BCM_GETTEXTMARGIN';
  Name[BCM_SETDROPDOWNSTATE] := 'BCM_SETDROPDOWNSTATE';
  Name[BCM_SETSPLITINFO] := 'BCM_SETSPLITINFO';
  Name[BCM_GETSPLITINFO] := 'BCM_GETSPLITINFO';
  Name[BCM_SETNOTE] := 'BCM_SETNOTE';
  Name[BCM_GETNOTE] := 'BCM_GETNOTE';
  Name[BCM_GETNOTELENGTH] := 'BCM_GETNOTELENGTH';
  Name[BCM_SETSHIELD] := 'BCM_SETSHIELD';
  Name[EM_SETCUEBANNER] := 'EM_SETCUEBANNER';
  Name[EM_GETCUEBANNER] := 'EM_GETCUEBANNER';
  Name[EM_SHOWBALLOONTIP] := 'EM_SHOWBALLOONTIP';
  Name[EM_HIDEBALLOONTIP] := 'EM_HIDEBALLOONTIP';
  Name[EM_SETHILITE] := 'EM_SETHILITE';
  Name[EM_GETHILITE] := 'EM_GETHILITE';
  Name[EM_NOSETFOCUS] := 'EM_NOSETFOCUS';
  Name[EM_TAKEFOCUS] := 'EM_TAKEFOCUS';
  Name[DL_BEGINDRAG] := 'DL_BEGINDRAG';
  Name[DL_DRAGGING] := 'DL_DRAGGING';
  Name[DL_DROPPED] := 'DL_DROPPED';
  Name[DL_CANCELDRAG] := 'DL_CANCELDRAG';
  Name[CM_ACTIVATE] := 'CM_ACTIVATE';
  Name[CM_DEACTIVATE] := 'CM_DEACTIVATE';
  Name[CM_GOTFOCUS] := 'CM_GOTFOCUS';
  Name[CM_LOSTFOCUS] := 'CM_LOSTFOCUS';
  Name[CM_CANCELMODE] := 'CM_CANCELMODE';
  Name[CM_DIALOGKEY] := 'CM_DIALOGKEY';
  Name[CM_DIALOGCHAR] := 'CM_DIALOGCHAR';
  Name[CM_FOCUSCHANGED] := 'CM_FOCUSCHANGED';
  Name[CM_PARENTFONTCHANGED] := 'CM_PARENTFONTCHANGED';
  Name[CM_PARENTCOLORCHANGED] := 'CM_PARENTCOLORCHANGED';
  Name[CM_HITTEST] := 'CM_HITTEST';
  Name[CM_VISIBLECHANGED] := 'CM_VISIBLECHANGED';
  Name[CM_ENABLEDCHANGED] := 'CM_ENABLEDCHANGED';
  Name[CM_COLORCHANGED] := 'CM_COLORCHANGED';
  Name[CM_FONTCHANGED] := 'CM_FONTCHANGED';
  Name[CM_CURSORCHANGED] := 'CM_CURSORCHANGED';
  Name[CM_CTL3DCHANGED] := 'CM_CTL3DCHANGED';
  Name[CM_PARENTCTL3DCHANGED] := 'CM_PARENTCTL3DCHANGED';
  Name[CM_TEXTCHANGED] := 'CM_TEXTCHANGED';
  Name[CM_MOUSEENTER] := 'CM_MOUSEENTER';
  Name[CM_MOUSELEAVE] := 'CM_MOUSELEAVE';
  Name[CM_MENUCHANGED] := 'CM_MENUCHANGED';
  Name[CM_APPKEYDOWN] := 'CM_APPKEYDOWN';
  Name[CM_APPSYSCOMMAND] := 'CM_APPSYSCOMMAND';
  Name[CM_BUTTONPRESSED] := 'CM_BUTTONPRESSED';
  Name[CM_SHOWINGCHANGED] := 'CM_SHOWINGCHANGED';
  Name[CM_ENTER] := 'CM_ENTER';
  Name[CM_EXIT] := 'CM_EXIT';
  Name[CM_DESIGNHITTEST] := 'CM_DESIGNHITTEST';
  Name[CM_ICONCHANGED] := 'CM_ICONCHANGED';
  Name[CM_WANTSPECIALKEY] := 'CM_WANTSPECIALKEY';
  Name[CM_INVOKEHELP] := 'CM_INVOKEHELP';
  Name[CM_WINDOWHOOK] := 'CM_WINDOWHOOK';
  Name[CM_RELEASE] := 'CM_RELEASE';
  Name[CM_SHOWHINTCHANGED] := 'CM_SHOWHINTCHANGED';
  Name[CM_PARENTSHOWHINTCHANGED] := 'CM_PARENTSHOWHINTCHANGED';
  Name[CM_SYSCOLORCHANGE] := 'CM_SYSCOLORCHANGE';
  Name[CM_WININICHANGE] := 'CM_WININICHANGE';
  Name[CM_FONTCHANGE] := 'CM_FONTCHANGE';
  Name[CM_TIMECHANGE] := 'CM_TIMECHANGE';
  Name[CM_TABSTOPCHANGED] := 'CM_TABSTOPCHANGED';
  Name[CM_UIACTIVATE] := 'CM_UIACTIVATE';
  Name[CM_UIDEACTIVATE] := 'CM_UIDEACTIVATE';
  Name[CM_DOCWINDOWACTIVATE] := 'CM_DOCWINDOWACTIVATE';
  Name[CM_CONTROLLISTCHANGE] := 'CM_CONTROLLISTCHANGE';
  Name[CM_GETDATALINK] := 'CM_GETDATALINK';
  Name[CM_CHILDKEY] := 'CM_CHILDKEY';
  Name[CM_DRAG] := 'CM_DRAG';
  Name[CM_HINTSHOW] := 'CM_HINTSHOW';
  Name[CM_DIALOGHANDLE] := 'CM_DIALOGHANDLE';
  Name[CM_ISTOOLCONTROL] := 'CM_ISTOOLCONTROL';
  Name[CM_RECREATEWND] := 'CM_RECREATEWND';
  Name[CM_INVALIDATE] := 'CM_INVALIDATE';
  Name[CM_SYSFONTCHANGED] := 'CM_SYSFONTCHANGED';
  Name[CM_CONTROLCHANGE] := 'CM_CONTROLCHANGE';
  Name[CM_CHANGED] := 'CM_CHANGED';
  Name[CM_DOCKCLIENT] := 'CM_DOCKCLIENT';
  Name[CM_UNDOCKCLIENT] := 'CM_UNDOCKCLIENT';
  Name[CM_FLOAT] := 'CM_FLOAT';
  Name[CM_BORDERCHANGED] := 'CM_BORDERCHANGED';
  Name[CM_BIDIMODECHANGED] := 'CM_BIDIMODECHANGED';
  Name[CM_PARENTBIDIMODECHANGED] := 'CM_PARENTBIDIMODECHANGED';
  Name[CM_ALLCHILDRENFLIPPED] := 'CM_ALLCHILDRENFLIPPED';
  Name[CM_ACTIONUPDATE] := 'CM_ACTIONUPDATE';
  Name[CM_ACTIONEXECUTE] := 'CM_ACTIONEXECUTE';
  Name[CM_HINTSHOWPAUSE] := 'CM_HINTSHOWPAUSE';
  Name[CM_DOCKNOTIFICATION] := 'CM_DOCKNOTIFICATION';
  Name[CM_MOUSEWHEEL] := 'CM_MOUSEWHEEL';
  Name[CM_ISSHORTCUT] := 'CM_ISSHORTCUT';
  Name[CM_UPDATEACTIONS] := 'CM_UPDATEACTIONS';
  Name[CM_INVALIDATEDOCKHOST] := 'CM_INVALIDATEDOCKHOST';
  Name[CM_SETACTIVECONTROL] := 'CM_SETACTIVECONTROL';
  Name[CM_POPUPHWNDDESTROY] := 'CM_POPUPHWNDDESTROY';
  Name[CM_CREATEPOPUP] := 'CM_CREATEPOPUP';
  Name[CM_DESTROYHANDLE] := 'CM_DESTROYHANDLE';
  Name[CM_MOUSEACTIVATE] := 'CM_MOUSEACTIVATE';
  Name[CM_CONTROLLISTCHANGING] := 'CM_CONTROLLISTCHANGING';
  Name[CM_BUFFEREDPRINTCLIENT] := 'CM_BUFFEREDPRINTCLIENT';
  Name[CM_UNTHEMECONTROL] := 'CM_UNTHEMECONTROL';
  Name[CM_DOUBLEBUFFEREDCHANGED] := 'CM_DOUBLEBUFFEREDCHANGED';
  Name[CM_PARENTDOUBLEBUFFEREDCHANGED] := 'CM_PARENTDOUBLEBUFFEREDCHANGED';
  Name[CM_STYLECHANGED] := 'CM_STYLECHANGED';
  Name[CM_GESTURE] := 'CM_GESTURE';
  Name[CM_CUSTOMGESTURESCHANGED] := 'CM_CUSTOMGESTURESCHANGED';
  Name[CM_GESTUREMANAGERCHANGED] := 'CM_GESTUREMANAGERCHANGED';
  Name[CM_STANDARDGESTURESCHANGED] := 'CM_STANDARDGESTURESCHANGED';
  Name[CM_INPUTLANGCHANGE] := 'CM_INPUTLANGCHANGE';
  Name[CM_TABLETOPTIONSCHANGED] := 'CM_TABLETOPTIONSCHANGED';
  Name[CM_PARENTTABLETOPTIONSCHANGED] := 'CM_PARENTTABLETOPTIONSCHANGED';
  Name[CM_CUSTOMSTYLECHANGED] := 'CM_CUSTOMSTYLECHANGED';
  Name[CM_SYSFONTSALLCHANGED] := 'CM_SYSFONTSALLCHANGED';
  Name[CM_PARENTVISIBLECHANGED] := 'CM_PARENTVISIBLECHANGED';
  Name[CM_SYSCOMMAND] := 'CM_SYSCOMMAND';
  Name[CM_REMOTESESSIONSTATUSCHANGED] := 'CM_REMOTESESSIONSTATUSCHANGED';
  Name[CN_CHARTOITEM] := 'CN_CHARTOITEM';
  Name[CN_COMMAND] := 'CN_COMMAND';
  Name[CN_COMPAREITEM] := 'CN_COMPAREITEM';
  Name[CN_CTLCOLORBTN] := 'CN_CTLCOLORBTN';
  Name[CN_CTLCOLORDLG] := 'CN_CTLCOLORDLG';
  Name[CN_CTLCOLOREDIT] := 'CN_CTLCOLOREDIT';
  Name[CN_CTLCOLORLISTBOX] := 'CN_CTLCOLORLISTBOX';
  Name[CN_CTLCOLORMSGBOX] := 'CN_CTLCOLORMSGBOX';
  Name[CN_CTLCOLORSCROLLBAR] := 'CN_CTLCOLORSCROLLBAR';
  Name[CN_CTLCOLORSTATIC] := 'CN_CTLCOLORSTATIC';
  Name[CN_DELETEITEM] := 'CN_DELETEITEM';
  Name[CN_DRAWITEM] := 'CN_DRAWITEM';
  Name[CN_HSCROLL] := 'CN_HSCROLL';
  Name[CN_MEASUREITEM] := 'CN_MEASUREITEM';
  Name[CN_PARENTNOTIFY] := 'CN_PARENTNOTIFY';
  Name[CN_VKEYTOITEM] := 'CN_VKEYTOITEM';
  Name[CN_VSCROLL] := 'CN_VSCROLL';
  Name[CN_KEYDOWN] := 'CN_KEYDOWN';
  Name[CN_KEYUP] := 'CN_KEYUP';
  Name[CN_CHAR] := 'CN_CHAR';
  Name[CN_SYSKEYDOWN] := 'CN_SYSKEYDOWN';
  Name[CN_SYSCHAR] := 'CN_SYSCHAR';
  Name[CN_NOTIFY] := 'CN_NOTIFY';
end;

end.
