unit uTextPrinter;

{
   Package: TIU - TEXT INTEGRATION UTILITIES
   Date Created: Oct 23, 2006
   Site Name: xxxxxxxxxxxxxxxx
   Developers: zzzzzzzzzuser, SGT
   Description: Mobile Electronic Documentation
   Note: This unit requires XWB*1.1 and TIU*1*244 in order to run.
         Includes Template routines from CPRS
}

interface
uses windows, dialogs, sysutils, printers, classes, graphics, forms;

type
  TTextPrinter = class(TObject)
  public
    Document: string;
    Header: string;
    Title: string;
    Indent: real;
    function Print(PrinterUse: TPrinter; Font: TFont; BeginEndDoc: boolean = True; Copies: integer = 1): boolean;
    constructor Create();
  private
    function WordWrapDoc(ToWrap: string; MaxChars: integer): string;
  end;

implementation

{TTextPrinter}

function TTextPrinter.WordWrapDoc(ToWrap: string; MaxChars: integer): string;
var
  TRet, TWordWrap, TScratch: TStringList;
  d, c, p: integer;
  sLine: string;
begin
  //List to store our items
  TRet := TStringList.Create();
  //Scratch list
  TScratch := TStringList.Create();
  TScratch.Text := ToWrap;
  //For word wrap
  TWordWrap := TStringList.Create();

   //Go through each line
  for d := 0 to TScratch.Count - 1 do
  begin
            //Get the line
    sLine := TScratch[d];
            //Too long?
    if Length(sLine) > MaxChars then
    begin
                     //Yes.  Break this into smaller lines
      TWordWrap.Text := WrapText(sLine, #13, [' ', '.', ',', #9], MaxChars);
                     //Now add the lines in
      for c := 0 to TWordWrap.Count - 1 do
      begin
                              //Get the item
        sLine := TWordWrap[c];
                              //Still too long?
        if Length(sLine) > MaxChars then
        begin
                                       //Break this into pieces
          p := 1;
          while (p <> -1) do
          begin
                                                //Can copy?
            if p <= Length(sLine) then
            begin
              TRet.Add(Copy(sLine, p, MaxChars));
              p := p + MaxChars;
            end
            else
              p := -1;
          end;

        end
        else
          TRet.Add(sLine);
      end;
    end
    else
      TRet.Add(sLine);

  end;

  //Return it
  Result := TRet.Text;
  //Release
  TRet.Free;
  TWordWrap.Free;

end;

function TTextPrinter.Print(PrinterUse: TPrinter; Font: TFont; BeginEndDoc: boolean = True; Copies: integer = 1): boolean;
var
  Draw: TCanvas;
  PageCount, Line, FHeight, HHeight, CHeight, IWidth, IHeight, MaxChars, p, h, d, c: integer;
  RPCount: Real;
  PixelsPerInchVertical, PixelsPerInchHorizontal: integer;
  LinesPerPage: integer;
  THeader, TDocument: TStringList;
  SHeader: string;
begin
  //Validate our printer
  if not assigned(PrinterUse) then
  begin
    Result := False;
    Exit;
  end;

  //Create our Header
  THeader := TStringList.Create();
  THeader.Text := Self.Header;

  //Document
  TDocument := TStringList.Create();

  //Print our pages
  try
      //Copies
    PrinterUse.Copies := Copies;
      //Begin the document?
    if BeginEndDoc = True then PrinterUse.BeginDoc;


        //Document Information
    PrinterUse.Title := Self.Title;

      //Get our Canvas
    Draw := PrinterUse.Canvas;
      //Set our Font
    Draw.Font := Font;

      //Calculate our content, sizes, pages, etc.
        //Pixels per inch on this device
    PixelsPerInchVertical := GetDeviceCaps(PrinterUse.Handle, LOGPIXELSY);
    PixelsPerInchHorizontal := GetDeviceCaps(PrinterUse.Handle, LOGPIXELSX);

        //Get the font height
    FHeight := Draw.TextHeight(' ');

        //Indents
    IHeight := Round(Indent * PixelsPerInchVertical);
    IWidth := Round(Indent * PixelsPerInchHorizontal);

        //Maximum number of characters per line
    MaxChars := (PrinterUse.PageWidth - (IWidth * 2)) div Draw.TextWidth(' ');

        //Header Height
    HHeight := (FHeight * THeader.Count) + FHeight;

        //Content Area
    CHeight := PrinterUse.PageHeight - ((IHeight * 2) + HHeight);
          //Adjust to actual print that will fit
    CHeight := Trunc(CHeight / FHeight) * FHeight;

        //Word Wrap the document
    TDocument.Text := Self.WordWrapDoc(Self.Document, MaxChars);

      //Page Count & Page
    LinesPerPage := Trunc(CHeight / FHeight);

    RPCount := TDocument.Count / LinesPerPage;
    if Frac(RPCount) > 0 then
      RPCount := RPCount + 1;
    PageCount := Trunc(Int(RPCount));

      //Now print our page(s)
    c := 0;
    for p := 1 to PageCount do
    begin
            //Print a header
      Line := IHeight;
                     //Now print out our header
                        //Replace tokens
      SHeader := Self.Header;
      SHeader := StringReplace(SHeader, '&p', IntToStr(p), [rfReplaceAll]);
      SHeader := StringReplace(SHeader, '&P', IntToStr(PageCount), [rfReplaceAll]);
                        //Set it
      THeader.Text := SHeader;
                        //Now print it
      for h := 0 to THeader.Count - 1 do
      begin
                                 //Output
        Draw.TextOut(IWidth, Line, THeader.Strings[h]);
                                 //Increment
        Line := Line + FHeight;
      end;

                     //Line after our header
      Draw.MoveTo(IWidth, Line);
      Draw.LineTo(PrinterUse.PageWidth - IWidth, Line);
                     //Increment
      Line := Line + FHeight;

            //Now render them
      for d := c to c + (LinesPerPage - 1) do
      begin
                  //Greater than our count?
        if d <= TDocument.Count - 1 then
        begin
                        //No.  Valid line
          Draw.TextOut(IWidth, Line, TDocument.Strings[d]);
          Line := Line + FHeight;
                        //Store the last line used
          c := d + 1;
        end;
      end;

            //Last line?
      if c < TDocument.Count then
      begin
                  //No.  We'll start a new page
        PrinterUse.NewPage;
      end;

    end;

      //Finish?
    if BeginEndDoc = True then PrinterUse.EndDoc;
      //Done
    Result := True;
  except
      //Problem printing document
    PrinterUse.Abort;
    Result := False;
    Application.HandleException(Self);
  end;

  //Cleanup
  TDocument.Free;
  THeader.Free;

end;

constructor TTextPrinter.Create();
begin
  Self.Indent := 0.5;
  Self.Header := 'Page &p of &P';
  inherited Create();
end;

end.
